#
# This file is part of the Cardinal Optimizer, all rights reserved.
#

import coptpy as cp
from coptpy import COPT

import itertools

# Optimization data for transportation problem
buildlimit    = 8

ncity         = 30
cities        = ['city' + str(i) for i in range(ncity)]

supply_list   = [19, 13, 20, 18, 23, 25, 22, 28, 18, 16, 17, 14, 22, 13, 18,
                 20, 15, 23, 25, 19, 18, 19, 22, 22, 21, 19, 17, 21, 15, 25]
supply        = dict(zip(cities, supply_list))

demand_list   = [4, 1, 2, 7, 9, 9, 3, 9, 5, 2, 4, 1, 9, 3, 1,
                 7, 5, 9, 2, 4, 9, 6, 5, 4, 5, 6, 8, 9, 5, 6]
demand        = dict(zip(cities, demand_list))

shipcost_list = [0, 61,35, 35,53,16,11, 58,28,34, 43,18, 48,30, 68,15,27, 53,26, 32,13,41,54,36,51, 68,29,20, 49, 48,
                 61, 0,96, 88,29,77,50, 92,57,75, 48,47, 76,32, 77,74,34, 75,87, 68,70,38, 8,87,71, 82,39,78,104,106,
                 35,96, 0, 18,88,21,45, 72,58,53, 65,49, 51,66, 93,22,61, 76,14, 40,33,76,88,43,76, 91,64,17, 43, 33,
                 35,88,18,  0,87,27,42, 84,62,62, 51,42, 33,62,100,20,55, 85,26, 25,39,76,81,55,83,100,63,16, 59, 51,
                 53,29,88, 87, 0,67,45, 66,37,54, 62,47, 85,26, 49,68,35, 48,77, 73,57,14,27,67,44, 54,24,73, 85, 90,
                 16,77,21, 27,67, 0,27, 57,36,34, 55,32, 51,46, 73,10,43, 57,10, 37,12,54,69,29,57, 73,43,12, 38, 34, 
                 11,50,45, 42,45,27, 0, 64,29,40, 35, 8, 46,20, 68,24,16, 55,37, 31,23,35,43,44,52, 69,22,29, 60, 59,  
                 58,92,72, 84,66,57,64,  0,37,24, 99,72,106,65, 31,67,72, 19,60, 90,47,54,87,30,23, 26,56,69, 38, 49, 
                 28,57,58, 62,37,36,29, 37, 0,18, 62,36, 74,28, 40,41,34, 26,45, 59,24,23,51,30,24, 41,19,46, 48, 54,
                 34,75,53, 62,54,34,40, 24,18, 0, 75,48, 81,45, 41,43,50, 24,39, 66,23,40,69,15,24, 39,37,46, 32, 39,
                 43,48,65, 51,62,55,35, 99,62,75,  0,27, 29,39, 99,46,30, 88,63, 27,57,59,41,79,85,101,48,49, 92, 89,
                 18,47,49, 42,47,32, 8, 72,36,48, 27, 0, 40,21, 74,27,13, 62,42, 27,31,39,39,52,60, 77,26,31, 67, 65,
                 48,76,51, 33,85,51,46,106,74,81, 29,40,  0,60,114,41,51,100,55, 15,59,79,69,80,98,115,66,40, 89, 82,
                 30,32,66, 62,26,46,20, 65,28,45, 39,21, 60, 0, 60,44,10, 51,56, 48,38,19,25,55,48, 63,10,49, 72, 74,
                 68,77,93,100,49,73,68, 31,40,41, 99,74,114,60,  0,80,69, 17,79, 99,61,42,75,53,17,  6,51,85, 67, 77,
                 15,74,22, 20,68,10,24, 67,41,43, 46,27, 41,44, 80, 0,39, 65,17, 26,20,56,66,39,63, 80,44, 5, 48, 43,
                 27,34,61, 55,35,43,16, 72,34,50, 30,13, 51,10, 69,39, 0, 59,53, 39,38,29,27,58,56, 72,18,44, 74, 74,
                 53,75,76, 85,48,57,55, 19,26,24, 88,62,100,51, 17,65,59,  0,63, 85,45,37,70,37, 4, 15,42,69, 51, 60,
                 26,87,14, 26,77,10,37, 60,45,39, 63,42, 55,56, 79,17,53, 63, 0, 42,21,64,80,30,63, 78,53,15, 34, 27,
                 32,68,40, 25,73,37,31, 90,59,66, 27,27, 15,48, 99,26,39, 85,42,  0,44,65,61,65,83,100,52,27, 75, 69,
                 13,70,33, 39,57,12,23, 47,24,23, 57,31, 59,38, 61,20,38, 45,21, 44, 0,44,63,22,44, 61,34,24, 36, 36,
                 41,38,76, 76,14,54,35, 54,23,40, 59,39, 79,19, 42,56,29, 37,64, 65,44, 0,34,53,33, 46,13,61, 71, 76,
                 54, 8,88, 81,27,69,43, 87,51,69, 41,39, 69,25, 75,66,27, 70,80, 61,63,34, 0,80,67, 79,32,71, 97, 99,
                 36,87,43, 55,67,29,44, 30,30,15, 79,52, 80,55, 53,39,58, 37,30, 65,22,53,80, 0,38, 51,48,41, 18, 25,
                 51,71,76, 83,44,57,52, 23,24,24, 85,60, 98,48, 17,63,56,  4,63, 83,44,33,67,38, 0, 17,38,68, 53, 62,
                 68,82,91,100,54,73,69, 26,41,39,101,77,115,63,  6,80,72, 15,78,100,61,46,79,51,17,  0,54,84, 63, 73,
                 29,39,64, 63,24,43,22, 56,19,37, 48,26, 66,10, 51,44,18, 42,53, 52,34,13,32,48,38, 54, 0,49, 66, 69,
                 20,78,17, 16,73,12,29, 69,46,46, 49,31, 40,49, 85, 5,44, 69,15, 27,24,61,71,41,68, 84,49, 0, 48, 42,
                 49,104,43,59,85,38,60, 38,48,32, 92,67, 89,72, 67,48,74, 51,34, 75,36,71,97,18,53, 63,66,48,  0, 12,
                 48,106,33,51,90,34,59, 49,54,39, 89,65, 82,74, 77,43,74, 60,27, 69,36,76,99,25,62, 73,69,42, 12,  0]

shipcost      = dict(zip(itertools.product(cities, cities), shipcost_list))


# Parameters for Lagrangian-Relaxation
iterlimit = 10
samelimit = 3

isLB      = 0
nSame     = 0
LB        = 0.0
UB        = 0.0
scale     = 1.0
step      = 0.0
norm      = 0.0
mult      = {city: 0.0 for city in cities}
slack     = {city: 0.0 for city in cities}

# Create COPT environment
env = cp.Envr()

# Create the LB/UB problem
mtrnloc = env.createModel()

# Disable log information
mtrnloc.setParam(COPT.Param.Logging, 0)

# Add variables to the LB/UB problem
vbuild = mtrnloc.addVars(cities, vtype=COPT.BINARY, nameprefix='build')
vship  = mtrnloc.addVars(cities, cities, ub=list(itertools.repeat(demand_list, ncity)), vtype=COPT.INTEGER, nameprefix='ship')

# Add constraints to the UB problem
csupply = mtrnloc.addConstrs((vship.sum(i, '*') <= supply[i] * vbuild[i] for i in cities), nameprefix='supply')
crelax  = mtrnloc.addConstrs((vship.sum('*', j) >= demand[j] for j in cities), nameprefix='relax')
climit  = mtrnloc.addConstr(vbuild.sum('*') <= buildlimit, name='limit')

# Set the objective function for the UB problem
mtrnloc.setObjective(vship.prod(shipcost), COPT.MINIMIZE)

# Solve the relaxed UB problem and initialize 'LB' with relaxed objective value
mtrnloc.solveLP()
LB = mtrnloc.lpobjval

# Initialize 'UB'
shipcost_tp = cp.tupledict(shipcost)
for i in cities:
  UB += max(shipcost_tp.select(i, '*'))

# Lagrangian-Relaxation loop
print("             *** Lagrangian-Relaxation Loop ***             ")
print("  Iter        LB               UB          scale        step")
for itercnt in range(iterlimit):
  # We need to solve the LB problem
  if isLB == 0:
    isLB = 1
    mtrnloc.remove(crelax)

  # Set the objective functhion for the LB problem
  mtrnloc.setObjective(vship.prod(shipcost) + \
                       cp.quicksum(mult[j] * (demand[j] - vship.sum('*', j)) for j in cities), \
                       COPT.MINIMIZE)

  # Solve the LB problem
  mtrnloc.solve()

  # Calculate 'slack'
  for j in cities:
    slack[j] = sum(vship[i, j].x for i in cities) - demand[j]

  # Improve 'LB'
  if mtrnloc.objval > LB + 1e-6:
    LB = mtrnloc.objval
    nSame = 0
  else:
    nSame += 1

  # Update 'scale' if no improvement in 'iterlimit' iterations
  if nSame == samelimit:
    scale /= 2.0
    nSame = 0

  # Calculate 'norm'
  norm = sum(slack[i]**2.0 for i in cities)

  # Update 'step'
  step = scale * (UB - mtrnloc.objval) / norm

  # Update 'mult'
  for i in cities:
    if mult[i] > step * slack[i]:
      mult[i] -= step * slack[i]
    else:
      mult[i] = 0.0

  # Test if we need to solve the UB problem
  sumsupply = sum(supply[i] * vbuild[i].x for i in cities)
  sumdemand = sum(demand.values())

  if sumsupply - sumdemand >= 1e-6:
    # We will solve the UB problem
    isLB = 0

    # Retrieve solution for the LB problem and fix the variables
    buildsol = mtrnloc.getInfo(COPT.Info.Value, vbuild)

    mtrnloc.setInfo(COPT.Info.LB, vbuild, buildsol)
    mtrnloc.setInfo(COPT.Info.UB, vbuild, buildsol)

    # Add relaxed constraints to the UB problem
    crelax = mtrnloc.addConstrs((vship.sum('*', j) >= demand[j] for j in cities), nameprefix='relax')

    # Set the objective function for the UB problem
    mtrnloc.setObjective(vship.prod(shipcost), COPT.MINIMIZE)

    # Solve the UB problem
    mtrnloc.solve()

    # Update 'UB'
    UB = min(UB, mtrnloc.objval)

    # Reset variables to initial bounds
    mtrnloc.setInfo(COPT.Info.LB, vbuild, 0.0)
    mtrnloc.setInfo(COPT.Info.UB, vbuild, 1.0)

  # Print statistics
  print('  {0:3d}    {1:12.6f}    {2:12.6f}    {3:8.6f}    {4:8.6f}'.format(itercnt, LB, UB, scale, step))
print("                      *** End Loop ***                      ")
