'
' This file is part of the Cardinal Optimizer, all rights reserved.
'
Imports Copt

Imports System

Class Socp_ex1
  Shared Sub Solve_SOC()
    Try
      ' Create COPT environment and model
      Dim env As New Envr()
      Dim model As Model = env.CreateModel("solve_soc")

      ' Add variables
      '
      '   minimize: z
      '
      '   bnds:
      '     x, y, t free, z non-negative
      '
      Dim infinity As Double = Copt.Consts.INFINITY

      Dim x As Var = model.AddVar(-infinity, infinity, 0.0, Copt.Consts.CONTINUOUS, "x")
      Dim y As Var = model.AddVar(-infinity, infinity, 0.0, Copt.Consts.CONTINUOUS, "y")
      Dim z As Var = model.AddVar(0, infinity, 0.0, Copt.Consts.CONTINUOUS, "z")
      Dim t As Var = model.AddVar(-infinity, infinity, 0.0, Copt.Consts.CONTINUOUS, "t")

      ' Set objective: z
      model.SetObjective(z, Copt.Consts.MINIMIZE)

      ' Add constraints
      '
      '   r0: 3*x + y >= 1
      '   c0: z^2 >= x^2 + 2*y^2
      '
      ' c0 is converted to:
      '
      '   r1: sqrt(2.0)*y - t = 0
      '   c1: z^2 >= x^2 + t^2
      '
      model.AddConstr(3.0 * x + y >= 1.0, "r0")
      model.AddConstr(Math.Sqrt(2.0) * y - t = 0.0, "r1")

      Dim cvars As VarArray = New VarArray()
      cvars.PushBack(z)
      cvars.PushBack(x)
      cvars.PushBack(t)
      model.AddCone(cvars, Copt.Consts.CONE_QUAD)

      ' Set parameters
      model.SetDblParam(Copt.DblParam.TimeLimit, 10)

      ' Solve the model
      model.Solve()

      ' Analyze solution
      If model.GetIntAttr(Copt.IntAttr.LpStatus) = COPT.Status.OPTIMAL Then
        Console.WriteLine(vbLf & "Found optimal solution:")
        Dim vars As VarArray = model.GetVars()

        For i As Integer = 0 To vars.Size() - 1
          Dim var As Var = vars.GetVar(i)
          Console.WriteLine("  " & var.GetName() & " = " & var.Get(Copt.DblInfo.Value))
        Next

        Console.WriteLine("Obj = " & model.GetDblAttr(Copt.DblAttr.LpObjVal))
      End If
    Catch e As CoptException
      Console.WriteLine("Error Code = " & e.GetCode())
    End Try
  End Sub

  Shared Sub Solve_RSOC()
    Try
      ' Create COPT environment and model
      Dim env As New Envr()
      Dim model As Model = env.CreateModel("solve_rsoc")

      ' Add variables
      '
      '   minimize: 1.5*x - 2*y + z
      '
      '   bnds:
      '     0 <= x <= 20
      '     y, z, r >= 0
      '     s, t free
      '
      Dim infinity As Double = Copt.Consts.INFINITY

      Dim x As Var = model.AddVar(0.0, 20.0, 0.0, Copt.Consts.CONTINUOUS, "x")
      Dim y As Var = model.AddVar(0.0, infinity, 0.0, Copt.Consts.CONTINUOUS, "y")
      Dim z As Var = model.AddVar(0.0, infinity, 0.0, Copt.Consts.CONTINUOUS, "z")
      Dim r As Var = model.AddVar(0.0, infinity, 0.0, Copt.Consts.CONTINUOUS, "r")
      Dim s As Var = model.AddVar(-infinity, infinity, 0.0, Copt.Consts.CONTINUOUS, "s")
      Dim t As Var = model.AddVar(-infinity, infinity, 0.0, Copt.Consts.CONTINUOUS, "t")

      ' Set objective: 1.5*x - 2*y + z
      model.SetObjective(1.5*x - 2*y + z, Copt.Consts.MINIMIZE)

      ' Add constraints
      '
      '   r0: 2*x + y >= 2
      '   r1: -x + 2*y <= 6
      '   r2: r = 1
      '   r3: 2.8284271247 * x + 0.7071067811 * y - s = 0
      '   r4: 3.0822070014 * y - t = 0
      '   c0: 2*z*r >= s^2 + t^2
      '
      model.AddConstr(2*x + y >= 2, "r0")
      model.AddConstr(-x + 2 * y <= 6, "r1")
      model.AddConstr(r = 1, "r2")
      model.AddConstr(2.8284271247 * x + 0.7071067811 * y - s = 0, "r3")
      model.AddConstr(3.0822070014 * y - t = 0, "r4")

      Dim rvars As VarArray = New VarArray()
      rvars.PushBack(z)
      rvars.PushBack(r)
      rvars.PushBack(s)
      rvars.PushBack(t)
      model.AddCone(rvars, Copt.Consts.CONE_RQUAD)

      ' Set parameters
      model.SetDblParam(Copt.DblParam.TimeLimit, 10)

      ' Solve the model
      model.Solve()

      ' Analyze solution
      If model.GetIntAttr(Copt.IntAttr.LpStatus) = COPT.Status.OPTIMAL Then
        Console.WriteLine(vbLf & "Found optimal solution:")
        Dim vars As VarArray = model.GetVars()

        For i As Integer = 0 To vars.Size() - 1
          Dim var As Var = vars.GetVar(i)
          Console.WriteLine("  " & var.GetName() & " = " & var.Get(Copt.DblInfo.Value))
        Next

        Console.WriteLine("Obj = " & model.GetDblAttr(Copt.DblAttr.LpObjVal))
      End If
    Catch e As CoptException
      Console.WriteLine("Error Code = " & e.GetCode())
    End Try
  End Sub

  Shared Sub Solve_MPS(filename As String)
    Try
      ' Create COPT environment and model
      Dim env As New Envr()
      Dim model As Model = env.CreateModel("solve_mps")

      ' Read SOCP from MPS file
      model.ReadMps(filename)

      ' Set parameters
      model.SetDblParam(Copt.DblParam.TimeLimit, 10)

      ' Solve the model
      model.Solve()
    Catch e As CoptException
      Console.WriteLine("Error Code = " & e.GetCode())
    End Try
  End Sub

  Shared Sub Main(ByVal cmdArgs() As String)
    ' Solve SOCP with regular cone
    Solve_SOC()

    ' Solve SOCP with rotated cone
    Solve_RSOC()

    ' Solve SOCP from MPS file
    If cmdArgs.Length > 0 Then
      Solve_MPS(cmdArgs(0))
    End If
  End Sub
End Class
