# Copyright (C), Cardinal Operations

from types import TracebackType
from typing import (
    Any,
    Callable,
    Dict,
    Generator,
    List,
    Iterable,
    Mapping,
    Optional,
    Sequence,
    Tuple,
    Type,
    TypeVar,
    Union,
    overload,
)
from scipy import sparse
import numpy as np

_T = TypeVar("_T")
_U = TypeVar("_U")
_Scalar = Union[int, float, str]
_LinExprLike = Union[float, Var, ExprBuilder, LinExpr]
_NlExprLike = Union[float, Var, ExprBuilder, LinExpr, QuadExpr, NlExpr]
_PsdExprLike = Union[float, Var, ExprBuilder, LinExpr, PsdExpr, PsdVar]
_QuadExprLike = Union[float, Var, ExprBuilder, LinExpr, QuadExpr]
_ModelComponent = Union[
    Cone,
    Var,
    PsdVar,
    Constraint,
    NlConstraint,
    LmiConstraint,
    QConstraint,
    PsdConstraint,
    GenConstr,
    SOS,
]

@overload
def multidict(
    __data: Mapping[_T, float],
) -> Tuple[tuplelist[_T], tupledict[_T, float]]: ...
@overload
def multidict(__data: Mapping[Any, Iterable[float]]) -> Tuple[Any, ...]: ...
@overload
def multidict(__data: Iterable[Tuple[Any, Iterable[float]]]) -> Tuple[Any, ...]: ...
@overload
def quicksum(termlist: Mapping[Any, Var]) -> LinExpr: ...
@overload
def quicksum(termlist: Iterable[_LinExprLike]) -> LinExpr: ...
@overload
def quicksum(termlist: Iterable[_QuadExprLike]) -> QuadExpr: ...
@overload
def psdquicksum(termlist: Mapping[Any, PsdVar]) -> PsdExpr: ...
@overload
def psdquicksum(termlist: Iterable[_LinExprLike]) -> LinExpr: ...
@overload
def psdquicksum(termlist: Iterable[_PsdExprLike]) -> PsdExpr: ...

class ILogCallback: ...

class LogCallback(ILogCallback):
    def __init__(self) -> None: ...
    def Display(self, msg: str) -> None: ...

class ICallback: ...

class CallbackBase(ICallback):
    def __init__(self) -> None: ...
    def callback(self) -> None: ...
    def interrupt(self) -> None: ...
    @overload
    def addUserCut(
        self, lhs: _LinExprLike, sense: str, rhs: _LinExprLike, name: str = ""
    ) -> None: ...
    @overload
    def addUserCut(self, lhs: ConstrBuilder, name: str = "") -> None: ...
    @overload
    def addUserCut(self, lhs: ConstrBuilderArray, name: str = "") -> None: ...
    @overload
    def addUserCuts(self, generator: ConstrBuilderArray) -> None: ...
    @overload
    def addUserCuts(self, generator: Iterable[ConstrBuilder]) -> None: ...
    @overload
    def addUserCuts(self, generator: Generator[ConstrBuilder, None, None]) -> None: ...
    @overload
    def addLazyConstr(
        self, lhs: _LinExprLike, sense: str, rhs: _LinExprLike, name: str = ""
    ) -> None: ...
    @overload
    def addLazyConstr(self, lhs: ConstrBuilder, name: str = "") -> None: ...
    @overload
    def addLazyConstr(self, lhs: ConstrBuilderArray, name: str = "") -> None: ...
    @overload
    def addLazyConstrs(self, generator: ConstrBuilderArray) -> None: ...
    @overload
    def addLazyConstrs(self, generator: Iterable[ConstrBuilder]) -> None: ...
    @overload
    def addLazyConstrs(
        self, generator: Generator[ConstrBuilder, None, None]
    ) -> None: ...
    def getInfo(self, cbinfo: str) -> Union[int, float]: ...
    def getIncumbent(
        self, vars: Union[Var, VarArray, tupledict[Any, Var]]
    ) -> List[float]: ...
    def getRelaxSol(
        self, vars: Union[Var, VarArray, tupledict[Any, Var]]
    ) -> List[float]: ...
    def getSolution(
        self, vars: Union[Var, VarArray, tupledict[Any, Var]]
    ) -> List[float]: ...
    def setSolution(self, vars: Union[Var, VarArray], vals: List[float]) -> None: ...
    def loadSolution(self) -> None: ...
    def where(self) -> int: ...

class ParamEnum:
    # Double parameters
    TimeLimit: str = ...
    SolTimeLimit: str = ...
    MatrixTol: str = ...
    FeasTol: str = ...
    DualTol: str = ...
    IntTol: str = ...
    PDLPTol: str = ...
    NLPTol: str = ...
    RelGap: str = ...
    AbsGap: str = ...
    TuneTimeLimit: str = ...
    TuneTargetLimit: str = ...
    TuenTargetRelGap: str = ...
    MultiObjTimeLimit: str = ...

    # Multi-objective parameters
    MultiObjPriority: str = ...
    MultiObjWeight: str = ...
    MultiObjAbsTol: str = ...
    MultiObjRelTol: str = ...

    # Integer parameters
    Logging: str = ...
    LogToConsole: str = ...
    Presolve: str = ...
    Scaling: str = ...
    Dualize: str = ...
    LpMethod: str = ...
    GPUMode: str = ...
    GPUDevice: str = ...
    ReqFarkasRay: str = ...
    ReqSensitivity: str = ...
    DualPrice: str = ...
    DualPerturb: str = ...
    CutLevel: str = ...
    RootCutLevel: str = ...
    TreeCutLevel: str = ...
    RootCutRounds: str = ...
    NodeCutRounds: str = ...
    HeurLevel: str = ...
    RoundingHeurLevel: str = ...
    DivingHeurLevel: str = ...
    FAPHeurLevel: str = ...
    SubMipHeurLevel: str = ...
    StrongBranching: str = ...
    ConflictAnalysis: str = ...
    NodeLimit: str = ...
    MipTasks: str = ...
    BarHomogeneous: str = ...
    BarOrder: str = ...
    BarStart: str = ...
    BarIterLimit: str = ...
    NonConvex: str = ...
    NLPMuUpdate: str = ...
    NLPLinScale: str = ...
    NLPIterLimit: str = ...
    Threads: str = ...
    BarThreads: str = ...
    SimplexThreads: str = ...
    CrossoverThreads: str = ...
    Crossover: str = ...
    SDPMethod: str = ...
    IISMethod: str = ...
    FeasRelaxMode: str = ...
    MipStartMode: str = ...
    MipStartNodeLimit: str = ...
    TuneMethod: str = ...
    TuneMode: str = ...
    TuneMeasure: str = ...
    TunePermutes: str = ...
    TuneOutputLevel: str = ...
    MultiObjParamMode: str = ...
    LazyConstraints: str = ...
    LinearizeIndicators: str = ...
    LinearizeSos: str = ...

class AttrEnum:
    # Double attributes
    SolvingTime: str = ...
    ObjConst: str = ...
    LpObjVal: str = ...
    BestObj: str = ...
    BestBnd: str = ...
    BestGap: str = ...
    FeasRelaxObj: str = ...

    # Integer attributes
    MultiObjs: str = ...
    Cols: str = ...
    PsdCols: str = ...
    Rows: str = ...
    Elems: str = ...
    QElems: str = ...
    NLElems: str = ...
    PsdElems: str = ...
    SymMats: str = ...
    Bins: str = ...
    Ints: str = ...
    Soss: str = ...
    Cones: str = ...
    ExpCones: str = ...
    AffineCones: str = ...
    QConstrs: str = ...
    NLConstrs: str = ...
    PsdConstrs: str = ...
    LmiConstrs: str = ...
    Indicators: str = ...
    IISCols: str = ...
    IISRows: str = ...
    IISSOSs: str = ...
    IISIndicators: str = ...
    ObjSense: str = ...
    LpStatus: str = ...
    MipStatus: str = ...
    SimplexIter: str = ...
    BarrierIter: str = ...
    PDLPIter: str = ...
    NodeCnt: str = ...
    PoolSols: str = ...
    TuneResults: str = ...
    HasLpSol: str = ...
    HasDualFarkas: str = ...
    HasPrimalRay: str = ...
    HasBasis: str = ...
    HasMipSol: str = ...
    HasBranchFactor: str = ...
    HasQObj: str = ...
    HasNLObj: str = ...
    HasPsdObj: str = ...
    HasIIS: str = ...
    HasFeasRelaxSol: str = ...
    HasSensitivity: str = ...
    IsMIP: str = ...
    IsMinIIS: str = ...

class InfoEnum:
    # Double information
    Obj: str = ...
    LB: str = ...
    UB: str = ...
    Value: str = ...
    Slack: str = ...
    Dual: str = ...
    RedCost: str = ...
    DualFarkas: str = ...
    PrimalRay: str = ...
    RelaxLB: str = ...
    RelaxUB: str = ...
    RelaxValue: str = ...
    BranchFactor: str = ...
    SAObjLow: str = ...
    SAObjUp: str = ...
    SALBLow: str = ...
    SALBUp: str = ...
    SAUBLow: str = ...
    SAUBUp: str = ...

class CBInfoEnum:
    # Callback information
    BestObj: str = ...
    BestBnd: str = ...
    HasIncumbent: str = ...
    Incumbent: str = ...
    MipCandidate: str = ...
    MipCandObj: str = ...
    RelaxSolution: str = ...
    RelaxSolObj: str = ...
    NodeStatus: str = ...

class COPT:
    # Version numbers
    VERSION_MAJOR: int = ...
    VERSION_MINOR: int = ...
    VERSION_TECHNICAL: int = ...

    # Optimization directions
    MINIMIZE: int = ...
    MAXIMIZE: int = ...

    # Infinity value
    INFINITY: float = ...
    # Undefined value
    UNDEFINED: float = ...

    # Constraint types
    LESS_EQUAL: str = ...
    GREATER_EQUAL: str = ...
    EQUAL: str = ...
    FREE: str = ...
    RANGE: str = ...

    # Variable types
    CONTINUOUS: str = ...
    BINARY: str = ...
    INTEGER: str = ...

    # SOS-variable types
    SOS_TYPE1: int = ...
    SOS_TYPE2: int = ...

    # Cone types
    CONE_QUAD: int = ...
    CONE_RQUAD: int = ...

    # Exponential cone types
    EXPCONE_PRIMAL: int = ...
    EXPCONE_DUAL: int = ...

    # Basis status
    BASIS_LOWER: int = ...
    BASIS_BASIC: int = ...
    BASIS_UPPER: int = ...
    BASIS_SUPERBASIC: int = ...
    BASIS_FIXED: int = ...

    # Optimization status
    UNSTARTED: int = ...
    OPTIMAL: int = ...
    INFEASIBLE: int = ...
    UNBOUNDED: int = ...
    INF_OR_UNB: int = ...
    NUMERICAL: int = ...
    NODELIMIT: int = ...
    IMPRECISE: int = ...
    TIMEOUT: int = ...
    UNFINISHED: int = ...
    INTERRUPTED: int = ...
    ITERLIMIT: int = ...

    # Callback context
    CBCONTEXT_MIPRELAX: int = ...
    CBCONTEXT_MIPSOL: int = ...
    CBCONTEXT_MIPNODE: int = ...
    CBCONTEXT_INCUMBENT: int = ...

    # Envr config keywords
    CLIENT_CAFILE: str = ...
    CLIENT_CERTFILE: str = ...
    CLIENT_CERTKEYFILE: str = ...
    CLIENT_CLUSTER: str = ...
    CLIENT_FLOATING: str = ...
    CLIENT_PASSWORD: str = ...
    CLIENT_PORT: str = ...
    CLIENT_PRIORITY: str = ...
    CLIENT_WAITTIME: str = ...
    # envr config keywords for web cliense service
    CLIENT_WEBSERVER: str = ...
    CLIENT_WEBLICENSEID: str = ...
    CLIENT_WEBACESSKEY: str = ...
    CLIENT_WEBTOKENDURATION: str = ...

    Param: ParamEnum = ...
    param: ParamEnum = ...
    Attr: AttrEnum = ...
    attr: AttrEnum = ...
    Info: InfoEnum = ...
    info: InfoEnum = ...
    CBInfo: CBInfoEnum = ...
    CbInfo: CBInfoEnum = ...
    cbinfo: CBInfoEnum = ...

class CoptError(Exception):
    retcode: int = ...
    message: str = ...
    def __init__(self, retcode: int = 0, message: str = "") -> None: ...

class ParamSetting:
    def __init__(self, model: Model) -> None: ...
    def __getattr__(self, paramname: str) -> Union[int, float]: ...
    def __setattr__(self, paramname: str, newval: _Scalar) -> None: ...

class SOS:
    # mutable info or params
    iis: int = ...

    def getIdx(self) -> int: ...
    def setIIS(self) -> int: ...
    def remove(self) -> None: ...

    # read-only info or attributes
    @property
    def index(self) -> int: ...

class SOSArray:
    def __init__(
        self, soss: Optional[Union[SOS, Iterable[SOS], Mapping[Any, SOS]]] = None
    ) -> None: ...
    def __getitem__(self, idx: int) -> SOS: ...
    def __iter__(self) -> SOSArray: ...
    def __next__(self) -> SOS: ...
    def pushBack(self, sos: Union[SOS, Iterable[SOS], Mapping[Any, SOS]]) -> None: ...
    def getSOS(self, idx: _Scalar) -> SOS: ...
    def getSize(self) -> int: ...
    @property
    def size(self) -> int: ...

class SOSBuilder:
    def __init__(self) -> None: ...
    @overload
    def setBuilder(
        self,
        sostype: int,
        vars: Union[Var, VarArray, Iterable[Var]],
        weights: Optional[Iterable[float]] = None,
    ) -> None: ...
    @overload
    def setBuilder(
        self,
        sostype: int,
        vars: Mapping[_T, Var],
        weights: Optional[Mapping[_T, float]] = None,
    ) -> None: ...
    def getType(self) -> int: ...
    def getVar(self, idx: _Scalar) -> Var: ...
    def getVars(self, idx: Optional[Iterable[int]]) -> VarArray: ...
    def getWeight(self, idx: _Scalar) -> float: ...
    def getWeights(self, idx: Optional[Iterable[int]]) -> List[float]: ...
    def getSize(self) -> int: ...

    # read-only info or attributes
    @property
    def type(self) -> int: ...
    @property
    def vars(self) -> VarArray: ...
    @property
    def weights(self) -> List[float]: ...
    @property
    def size(self) -> int: ...

class SOSBuilderArray:
    def __init__(
        self,
        sosbuilders: Optional[
            Union[SOSBuilder, Iterable[SOSBuilder], Mapping[Any, SOSBuilder]]
        ] = None,
    ) -> None: ...
    def __getitem__(self, idx: int) -> SOSBuilder: ...
    def __iter__(self) -> SOSBuilderArray: ...
    def __next__(self) -> SOSBuilder: ...
    def pushBack(
        self,
        sosbuilder: Union[SOSBuilder, Iterable[SOSBuilder], Mapping[Any, SOSBuilder]],
    ) -> None: ...
    def getBuilder(self, idx: _Scalar) -> SOSBuilder: ...
    def getSize(self) -> int: ...
    @property
    def size(self) -> int: ...

class Cone:
    def __getattr__(self, attrname: str) -> int: ...
    def getIdx(self) -> int: ...
    def remove(self) -> None: ...

    # read-only info or attributes
    @property
    def index(self) -> int: ...

class ConeArray:
    def __init__(
        self, cones: Optional[Union[Cone, Iterable[Cone], Mapping[Any, Cone]]]
    ) -> None: ...
    def __getitem__(self, idx: int) -> Cone: ...
    def __iter__(self) -> ConeArray: ...
    def __next__(self) -> Cone: ...
    def pushBack(
        self, cone: Union[Cone, Iterable[Cone], Mapping[Any, Cone]]
    ) -> None: ...
    def getCone(self, idx: _Scalar) -> Cone: ...
    def getSize(self) -> int: ...
    @property
    def size(self) -> int: ...

class ConeBuilder:
    def __init__(self) -> None: ...
    def setBuilder(
        self,
        conetype: int,
        vars: Union[Var, VarArray, Iterable[Var], Mapping[Any, Var]],
    ) -> None: ...
    def getType(self) -> int: ...
    def getVar(self, idx: _Scalar) -> Var: ...
    def getVars(self, idx: Optional[Iterable[int]] = None) -> VarArray: ...
    def getSize(self) -> int: ...

    # read-only info or attributes
    @property
    def type(self) -> int: ...
    @property
    def vars(self) -> VarArray: ...
    @property
    def size(self) -> int: ...

class ConeBuilderArray:
    def __init__(
        self,
        conebuilders: Optional[
            Union[ConeBuilder, Iterable[ConeBuilder], Mapping[Any, ConeBuilder]]
        ] = None,
    ) -> None: ...
    def __getitem__(self, idx: int) -> ConeBuilder: ...
    def __iter__(self) -> ConeBuilderArray: ...
    def __next__(self) -> ConeBuilder: ...
    def pushBack(
        self,
        conebuilder: Union[
            ConeBuilder, Iterable[ConeBuilder], Mapping[Any, ConeBuilder]
        ],
    ) -> None: ...
    def getBuilder(self, idx: _Scalar) -> ConeBuilder: ...
    def getSize(self) -> int: ...
    @property
    def size(self) -> int: ...

class ExpCone:
    def __getattr__(self, attrname: str) -> int: ...
    def getIdx(self) -> int: ...
    def remove(self) -> None: ...

    # read-only info or attributes
    @property
    def index(self) -> int: ...

class ExpConeArray:
    def __init__(
        self, cones: Optional[Union[ExpCone, Iterable[ExpCone], Mapping[Any, ExpCone]]]
    ) -> None: ...
    def __getitem__(self, idx: int) -> ExpCone: ...
    def __iter__(self) -> ExpConeArray: ...
    def __next__(self) -> ExpCone: ...
    def pushBack(
        self, cone: Union[ExpCone, Iterable[ExpCone], Mapping[Any, ExpCone]]
    ) -> None: ...
    def getCone(self, idx: _Scalar) -> ExpCone: ...
    def getSize(self) -> int: ...
    @property
    def size(self) -> int: ...

class ExpConeBuilder:
    def __init__(self) -> None: ...
    def setBuilder(
        self,
        conetype: int,
        vars: Union[Var, VarArray, Iterable[Var], Mapping[Any, Var]],
    ) -> None: ...
    def getType(self) -> int: ...
    def getVar(self, idx: _Scalar) -> Var: ...
    def getVars(self, idx: Optional[Iterable[int]] = None) -> VarArray: ...
    def getSize(self) -> int: ...

    # read-only info or attributes
    @property
    def type(self) -> int: ...
    @property
    def vars(self) -> VarArray: ...
    @property
    def size(self) -> int: ...

class ExpConeBuilderArray:
    def __init__(
        self,
        conebuilders: Optional[
            Union[
                ExpConeBuilder, Iterable[ExpConeBuilder], Mapping[Any, ExpConeBuilder]
            ]
        ] = None,
    ) -> None: ...
    def __getitem__(self, idx: int) -> ExpConeBuilder: ...
    def __iter__(self) -> ExpConeBuilderArray: ...
    def __next__(self) -> ExpConeBuilder: ...
    def pushBack(
        self,
        conebuilder: Union[
            ExpConeBuilder, Iterable[ExpConeBuilder], Mapping[Any, ExpConeBuilder]
        ],
    ) -> None: ...
    def getBuilder(self, idx: _Scalar) -> ExpConeBuilder: ...
    def getSize(self) -> int: ...
    @property
    def size(self) -> int: ...

class AffineCone:
    # mutable info or params
    name: str = ...

    def __getattr__(self, attrname: str) -> int: ...
    def getIdx(self) -> int: ...
    def getName(self) -> str: ...
    def setName(self, newname: str) -> None: ...
    def remove(self) -> None: ...

    # read-only info or attributes
    @property
    def index(self) -> int: ...

class AffineConeArray:
    def __init__(
        self,
        cones: Optional[
            Union[AffineCone, Iterable[AffineCone], Mapping[Any, AffineCone]]
        ],
    ) -> None: ...
    def __getitem__(self, idx: int) -> AffineCone: ...
    def __iter__(self) -> AffineConeArray: ...
    def __next__(self) -> AffineCone: ...
    def pushBack(
        self, cone: Union[AffineCone, Iterable[AffineCone], Mapping[Any, AffineCone]]
    ) -> None: ...
    def getCone(self, idx: _Scalar) -> AffineCone: ...
    def getSize(self) -> int: ...
    @property
    def size(self) -> int: ...

class AffineConeBuilder:
    def __init__(self) -> None: ...
    def setBuilder(
        self,
        conetype: int,
        exprs: Union[
            MLinExpr,
            MPsdExpr,
            _PsdExprLike,
            Iterable[_PsdExprLike],
            Mapping[Any, _PsdExprLike],
        ],
    ) -> None: ...
    def getType(self) -> int: ...
    def getSize(self) -> int: ...
    def hasPsdTerm(self) -> bool: ...
    def getExpr(self, idx: _Scalar) -> Union[LinExpr, PsdExpr]: ...
    def getPsdExpr(self, idx: _Scalar) -> PsdExpr: ...
    def getPsdExprs(
        self, idx: Optional[Iterable[int]] = None
    ) -> Union[List[PsdExpr], MPsdExpr]: ...
    def getExprs(
        self, idx: Optional[Iterable[int]] = None
    ) -> Union[List[LinExpr], List[PsdExpr], MLinExpr, MPsdExpr]: ...

    # read-only info or attributes
    @property
    def type(self) -> int: ...
    @property
    def exprs(self) -> Union[List[LinExpr], List[PsdExpr], MLinExpr, MPsdExpr]: ...
    @property
    def size(self) -> int: ...

class AffineConeBuilderArray:
    def __init__(
        self,
        conebuilders: Optional[
            Union[
                AffineConeBuilder,
                Iterable[AffineConeBuilder],
                Mapping[Any, AffineConeBuilder],
            ]
        ] = None,
    ) -> None: ...
    def __getitem__(self, idx: int) -> AffineConeBuilder: ...
    def __iter__(self) -> AffineConeBuilderArray: ...
    def __next__(self) -> AffineConeBuilder: ...
    def pushBack(
        self,
        conebuilder: Union[
            AffineConeBuilder,
            Iterable[AffineConeBuilder],
            Mapping[Any, AffineConeBuilder],
        ],
    ) -> None: ...
    def getBuilder(self, idx: _Scalar) -> AffineConeBuilder: ...
    def getSize(self) -> int: ...
    @property
    def size(self) -> int: ...

class Var:
    # mutable info or params
    # TODO: add more if needed
    lb: float = ...
    LB: float = ...
    name: str = ...
    ub: float = ...
    UB: float = ...
    vtype: str = ...
    VType: str = ...

    def getType(self) -> str: ...
    def getName(self) -> str: ...
    def getBasis(self) -> int: ...
    def getLowerIIS(self) -> float: ...
    def getUpperIIS(self) -> float: ...
    def getIdx(self) -> int: ...
    def setType(self, newtype: str) -> None: ...
    def setName(self, newname: str) -> None: ...
    def getInfo(self, infoname: str) -> float: ...
    def setInfo(self, infoname: str, newval: float) -> None: ...
    def remove(self) -> None: ...

    # read-only info or attributes
    # TODO: add more if needed
    @property
    def basis(self) -> int: ...
    @property
    def index(self) -> int: ...
    @property
    def x(self) -> float: ...
    @property
    def X(self) -> float: ...
    @property
    def rc(self) -> float: ...
    @property
    def iislb(self) -> float: ...
    @property
    def iisub(self) -> float: ...
    @property
    def relaxlb(self) -> float: ...
    @property
    def relaxub(self) -> float: ...
    @property
    def size(self) -> int: ...
    @property
    def shape(self) -> int: ...
    @overload
    def __add__(self, other: Union[float, Var, ExprBuilder]) -> ExprBuilder: ...
    @overload
    def __add__(self, other: LinExpr) -> LinExpr: ...
    @overload
    def __add__(self, other: QuadExpr) -> QuadExpr: ...
    @overload
    def __add__(self, other: PsdExpr) -> PsdExpr: ...
    @overload
    def __add__(self, other: Union[MVar, MLinExpr]) -> MLinExpr: ...
    @overload
    def __add__(self, other: MQuadExpr) -> MQuadExpr: ...
    @overload
    def __radd__(self, other: Union[float, Var, ExprBuilder]) -> ExprBuilder: ...
    @overload
    def __radd__(self, other: LinExpr) -> LinExpr: ...
    @overload
    def __radd__(self, other: QuadExpr) -> QuadExpr: ...
    @overload
    def __radd__(self, other: PsdExpr) -> PsdExpr: ...
    @overload
    def __sub__(self, other: Union[float, Var, ExprBuilder]) -> ExprBuilder: ...
    @overload
    def __sub__(self, other: LinExpr) -> LinExpr: ...
    @overload
    def __sub__(self, other: QuadExpr) -> QuadExpr: ...
    @overload
    def __sub__(self, other: PsdExpr) -> PsdExpr: ...
    @overload
    def __sub__(self, other: Union[MVar, MLinExpr]) -> MLinExpr: ...
    @overload
    def __sub__(self, other: MQuadExpr) -> MQuadExpr: ...
    @overload
    def __rsub__(self, other: Union[float, Var, ExprBuilder]) -> ExprBuilder: ...
    @overload
    def __rsub__(self, other: LinExpr) -> LinExpr: ...
    @overload
    def __rsub__(self, other: QuadExpr) -> QuadExpr: ...
    @overload
    def __rsub__(self, other: PsdExpr) -> PsdExpr: ...
    @overload
    def __mul__(self, other: float) -> ExprBuilder: ...
    @overload
    def __mul__(self, other: Union[Var, ExprBuilder, LinExpr]) -> QuadExpr: ...
    @overload
    def __mul__(self, other: Union[MVar, MLinExpr]) -> MQuadExpr: ...
    @overload
    def __mul__(self, other: Union[SymMatrix, SymMatExpr]) -> LmiExpr: ...
    @overload
    def __rmul__(self, other: float) -> ExprBuilder: ...
    @overload
    def __rmul__(self, other: Union[Var, ExprBuilder, LinExpr]) -> QuadExpr: ...
    def __pow__(self, other: int) -> QuadExpr: ...
    def __truediv__(self, other: float) -> ExprBuilder: ...
    def __neg__(self) -> ExprBuilder: ...
    # '==', '<=' and '>=' are used as constraint sense
    @overload
    def __eq__(self, other: _LinExprLike) -> ConstrBuilder: ...
    @overload
    def __eq__(self, other: QuadExpr) -> QConstrBuilder: ...
    @overload
    def __eq__(self, other: PsdExpr) -> PsdConstrBuilder: ...
    @overload
    def __eq__(self, other: Union[MVar, MLinExpr]) -> MConstrBuilder: ...
    @overload
    def __eq__(self, other: MQuadExpr) -> MQConstrBuilder: ...
    @overload
    def __le__(self, other: _LinExprLike) -> ConstrBuilder: ...
    @overload
    def __le__(self, other: QuadExpr) -> QConstrBuilder: ...
    @overload
    def __le__(self, other: PsdExpr) -> PsdConstrBuilder: ...
    @overload
    def __le__(self, other: Union[MVar, MLinExpr]) -> MConstrBuilder: ...
    @overload
    def __le__(self, other: MQuadExpr) -> MQConstrBuilder: ...
    @overload
    def __ge__(self, other: _LinExprLike) -> ConstrBuilder: ...
    @overload
    def __ge__(self, other: QuadExpr) -> QConstrBuilder: ...
    @overload
    def __ge__(self, other: PsdExpr) -> PsdConstrBuilder: ...
    @overload
    def __ge__(self, other: Union[MVar, MLinExpr]) -> MConstrBuilder: ...
    @overload
    def __ge__(self, other: MQuadExpr) -> MQConstrBuilder: ...

class VarArray:
    def __init__(
        self, vars: Optional[Union[Var, Iterable[Var], Mapping[Any, Var]]] = None
    ) -> None: ...
    def __getitem__(self, idx: int) -> Var: ...
    def __iter__(self) -> VarArray: ...
    def __next__(self) -> Var: ...
    def pushBack(self, var: Union[Var, Iterable[Var], Mapping[Any, Var]]) -> None: ...
    def getVar(self, idx: int) -> Var: ...
    def getAll(self) -> List[Var]: ...
    def getSize(self) -> int: ...
    @property
    def size(self) -> int: ...

class PsdVar:
    name: str = ...

    def __getitem__(
        self, idx: Union[int, slice, Tuple[Union[int, slice], ...]]
    ) -> MPsdExpr: ...
    def diagonal(self, offset: int = 0) -> MPsdExpr: ...
    def pick(self, indexes: Union[List[int], List[List[int]], NdArray]) -> MPsdExpr: ...
    def sum(self) -> PsdExpr: ...
    def toexpr(self) -> MPsdExpr: ...
    def getName(self) -> str: ...
    def getIdx(self) -> int: ...
    def getDim(self) -> int: ...
    def getLen(self) -> int: ...
    def setName(self, newname: str) -> None: ...
    def getInfo(self, infoname: str) -> List[float]: ...
    def remove(self) -> None: ...
    @property
    def x(self) -> List[float]: ...
    @property
    def dual(self) -> List[float]: ...
    @property
    def rc(self) -> List[float]: ...
    @property
    def dim(self) -> int: ...
    @property
    def len(self) -> int: ...
    @property
    def index(self) -> int: ...
    @property
    def size(self) -> int: ...
    @property
    def shape(self) -> Tuple[int, int]: ...
    @overload
    def __mul__(self, mat: SymMatrix) -> PsdExpr: ...
    @overload
    def __mul__(self, mat: SymMatExpr) -> PsdExpr: ...
    @overload
    def __rmul__(self, mat: SymMatrix) -> PsdExpr: ...
    @overload
    def __rmul__(self, mat: SymMatExpr) -> PsdExpr: ...

class PsdVarArray:
    def __init__(
        self,
        vars: Optional[Union[PsdVar, Iterable[PsdVar], Mapping[Any, PsdVar]]] = None,
    ) -> None: ...
    def __getitem__(self, idx: int) -> PsdVar: ...
    def __iter__(self) -> PsdVarArray: ...
    def __next__(self) -> PsdVar: ...
    def pushBack(
        self, var: Union[PsdVar, Iterable[PsdVar], Mapping[Any, PsdVar]]
    ) -> None: ...
    def getPsdVar(self, idx: int) -> PsdVar: ...
    def getSize(self) -> int: ...
    def reserve(self, n: int) -> None: ...
    @property
    def size(self) -> int: ...

class SymMatExpr:
    def __init__(self, expr: SymMatrix = None) -> None: ...
    def setCoeff(self, idx: int, val: float) -> None: ...
    def getCoeff(self, idx: int) -> float: ...
    def getSymMat(self, idx: int) -> SymMatrix: ...
    def getDim(self) -> int: ...
    def getSize(self) -> int: ...
    def addTerm(self, mat: SymMatrix, coeff: float = 1.0) -> None: ...
    def addTerms(
        self,
        mats: Union[
            SymMatrix, SymMatrixArray, Iterable[SymMatrix], Mapping[Any, SymMatrix]
        ],
        coeffs: Union[float, Sequence[float]],
    ) -> None: ...
    def addSymMatExpr(self, expr: SymMatExpr, mult: float = 1.0) -> None: ...
    def clone(self) -> None: ...
    def reserve(self, n: int) -> None: ...
    def remove(self, item: Union[int, SymMatrix]) -> None: ...
    @property
    def size(self) -> int: ...
    @property
    def dim(self) -> int: ...
    def __add__(self, other: Union[SymMatrix, SymMatExpr]) -> SymMatExpr: ...
    def __radd__(self, other: Union[SymMatrix, SymMatExpr]) -> SymMatExpr: ...
    def __iadd__(self, other: Union[SymMatrix, SymMatExpr]) -> SymMatExpr: ...
    def __sub__(self, other: Union[SymMatrix, SymMatExpr]) -> SymMatExpr: ...
    def __rsub__(self, other: Union[SymMatrix, SymMatExpr]) -> SymMatExpr: ...
    def __isub__(self, other: Union[SymMatrix, SymMatExpr]) -> SymMatExpr: ...
    @overload
    def __mul__(self, other: float) -> SymMatExpr: ...
    @overload
    def __mul__(self, other: PsdVar) -> PsdExpr: ...
    @overload
    def __mul__(self, other: Var) -> LmiExpr: ...
    @overload
    def __rmul__(self, other: float) -> SymMatExpr: ...
    @overload
    def __rmul__(self, other: PsdVar) -> PsdExpr: ...
    def __imul__(self, other: float) -> SymMatExpr: ...
    def __truediv__(self, other: float) -> SymMatExpr: ...
    def __itruediv__(self, other: float) -> SymMatExpr: ...
    def __neg__(self) -> SymMatExpr: ...

class SymMatrix:
    def getIdx(self) -> int: ...
    def getDim(self) -> int: ...
    @property
    def index(self) -> int: ...
    def dim(self) -> int: ...
    def size(self) -> Tuple[int, int]: ...
    def shape(self) -> Tuple[int, int]: ...
    def __add__(self, other: Union[SymMatrix, SymMatExpr]) -> SymMatExpr: ...
    def __radd__(self, other: Union[SymMatrix, SymMatExpr]) -> SymMatExpr: ...
    def __sub__(self, other: Union[SymMatrix, SymMatExpr]) -> SymMatExpr: ...
    def __rsub__(self, other: Union[SymMatrix, SymMatExpr]) -> SymMatExpr: ...
    @overload
    def __mul__(self, other: float) -> SymMatExpr: ...
    @overload
    def __mul__(self, other: PsdVar) -> PsdExpr: ...
    @overload
    def __mul__(self, other: Var) -> LmiExpr: ...
    @overload
    def __rmul__(self, other: float) -> SymMatExpr: ...
    @overload
    def __rmul__(self, other: PsdVar) -> PsdExpr: ...
    def __truediv__(self, other: float) -> SymMatExpr: ...
    def __neg__(self) -> SymMatExpr: ...

class SymMatrixArray:
    def __init__(
        self,
        mats: Optional[
            Union[SymMatrix, Iterable[SymMatrix], Mapping[Any, SymMatrix]]
        ] = None,
    ) -> None: ...
    def __getitem__(self, idx: int) -> SymMatrix: ...
    def __iter__(self) -> SymMatrixArray: ...
    def __next__(self) -> SymMatrix: ...
    def pushBack(
        self, mat: Union[SymMatrix, Iterable[SymMatrix], Mapping[Any, SymMatrix]]
    ) -> None: ...
    def getMatrix(self, idx: int) -> SymMatrix: ...
    def getSize(self) -> int: ...
    def reserve(self, n: int) -> None: ...
    @property
    def size(self) -> int: ...

class Constraint:
    # mutable info or params
    # TODO: add more if needed
    name: str = ...

    def getName(self) -> str: ...
    def getBasis(self) -> int: ...
    def getLowerIIS(self) -> float: ...
    def getUpperIIS(self) -> float: ...
    def getIdx(self) -> int: ...
    def setName(self, newname: str) -> None: ...
    def getInfo(self, infoname: str) -> float: ...
    def setInfo(self, infoname: str, newval: float) -> None: ...
    def remove(self) -> None: ...

    # read-only info or attributes
    # TODO: add more if needed
    @property
    def basis(self) -> int: ...
    @property
    def index(self) -> int: ...
    @property
    def pi(self) -> float: ...
    @property
    def iislb(self) -> float: ...
    @property
    def iisub(self) -> float: ...
    @property
    def relaxlb(self) -> float: ...
    @property
    def relaxub(self) -> float: ...

class ConstrArray:
    def __init__(
        self,
        constrs: Optional[
            Union[Constraint, Iterable[Constraint], Mapping[Any, Constraint]]
        ] = None,
    ) -> None: ...
    def __getitem__(self, idx: int) -> Constraint: ...
    def __iter__(self) -> ConstrArray: ...
    def __next__(self) -> Constraint: ...
    def pushBack(
        self, constr: Union[Constraint, Iterable[Constraint], Mapping[Any, Constraint]]
    ) -> None: ...
    def getConstr(self, idx: int) -> Constraint: ...
    def getAll(self) -> List[Constraint]: ...
    def getSize(self) -> int: ...
    @property
    def size(self) -> int: ...

class ConstrBuilder:
    def __init__(self) -> None: ...
    def setBuilder(self, expr: _LinExprLike, sense: str, rhs: float) -> None: ...
    def getExpr(self) -> LinExpr: ...
    def getSense(self) -> str: ...
    def __rshift__(self, other: ConstrBuilder) -> GenConstrBuilder: ...

class ConstrBuilderArray:
    def __init__(
        self,
        constrs: Optional[
            Union[ConstrBuilder, Iterable[ConstrBuilder], Mapping[Any, ConstrBuilder]]
        ] = None,
    ) -> None: ...
    def __getitem__(self, idx: int) -> ConstrBuilder: ...
    def __iter__(self) -> ConstrBuilderArray: ...
    def __next__(self) -> ConstrBuilder: ...
    def pushBack(
        self,
        constrbuilder: Union[
            ConstrBuilder, Iterable[ConstrBuilder], Mapping[Any, ConstrBuilder]
        ],
    ) -> None: ...
    def getBuilder(self, idx: int) -> ConstrBuilder: ...
    def getSize(self) -> int: ...
    @property
    def size(self) -> int: ...

class GenConstrX:
    def __init__(self, type: int, name: str) -> None: ...
    def getAttr(self, attrname: str) -> Union[int, str]: ...
    def setAttr(self, attrname: str, newval: str) -> None: ...

class GenConstr:
    name: str = ...

    def getIdx(self) -> int: ...
    def getIIS(self) -> int: ...
    def getName(self) -> str: ...
    def remove(self) -> None: ...
    def setName(self, newname: str) -> None: ...

    # read-only info or attributes
    @property
    def index(self) -> int: ...
    @property
    def name(self) -> str: ...
    @property
    def iis(self) -> int: ...

class GenConstrArray:
    def __init__(
        self,
        constrs: Optional[
            Union[GenConstr, Iterable[GenConstr], Mapping[Any, GenConstr]]
        ] = None,
    ) -> None: ...
    def __getitem__(self, idx: int) -> GenConstr: ...
    def __iter__(self) -> GenConstrArray: ...
    def __next__(self) -> GenConstr: ...
    def pushBack(
        self, genconstr: Union[GenConstr, Iterable[GenConstr], Mapping[Any, GenConstr]]
    ) -> None: ...
    def getGenConstr(self, idx: int) -> GenConstr: ...
    def getSize(self) -> int: ...
    def reserve(self, n: int) -> None: ...
    def tolist(self) -> List[GenConstr]: ...
    @property
    def size(self) -> int: ...

class GenConstrBuilder:
    def __init__(self) -> None: ...
    def setBuilder(
        self,
        var: Var,
        val: int,
        expr: _LinExprLike,
        sense: str,
        type: int = COPT.INDICATOR_IF,
    ) -> None: ...
    def getBinVar(self) -> Var: ...
    def getBinVal(self) -> int: ...
    def getExpr(self) -> LinExpr: ...
    def getSense(self) -> str: ...
    def getIndType(self) -> int: ...

    # read-only info or attributes
    @property
    def binvar(self) -> Var: ...
    @property
    def binval(self) -> int: ...
    @property
    def expr(self) -> LinExpr: ...
    @property
    def sense(self) -> str: ...
    @property
    def type(self) -> int: ...

class GenConstrBuilderArray:
    def __init__(
        self,
        constrs: Optional[
            Union[
                GenConstrBuilder,
                Iterable[GenConstrBuilder],
                Mapping[Any, GenConstrBuilder],
            ]
        ] = None,
    ) -> None: ...
    def __getitem__(self, idx: int) -> GenConstrBuilder: ...
    def __iter__(self) -> GenConstrBuilderArray: ...
    def __next__(self) -> GenConstrBuilder: ...
    def pushBack(
        self,
        genconstrbuilder: Union[
            GenConstrBuilder, Iterable[GenConstrBuilder], Mapping[Any, GenConstrBuilder]
        ],
    ) -> None: ...
    def getBuilder(self, idx: int) -> GenConstrBuilder: ...
    def getSize(self) -> int: ...
    @property
    def size(self) -> int: ...

class NlConstraint:
    name: str = ...

    def getName(self) -> str: ...
    def getIdx(self) -> int: ...
    def setName(self, newname: str) -> None: ...
    def getRhs(self) -> float: ...
    def getSense(self) -> str: ...
    def setRhs(self, rhs: float) -> None: ...
    def setSense(self, sense: str) -> None: ...
    def getInfo(self, infoname: str) -> float: ...
    def setInfo(self, infoname: str, newval: float) -> None: ...
    def remove(self) -> None: ...

class NlConstrArray:
    def __init__(
        self,
        constrs: Optional[
            Union[NlConstraint, Iterable[NlConstraint], Mapping[Any, NlConstraint]]
        ] = None,
    ) -> None: ...
    def __getitem__(self, idx: int) -> NlConstraint: ...
    def __iter__(self) -> NlConstrArray: ...
    def __next__(self) -> NlConstraint: ...
    def pushBack(
        self,
        qconstrs: Union[
            NlConstraint, Iterable[NlConstraint], Mapping[Any, NlConstraint]
        ],
    ) -> None: ...
    def getNlConstr(self, idx: int) -> NlConstraint: ...
    def getSize(self) -> int: ...
    @property
    def size(self) -> int: ...

class NlConstrBuilder:
    def __init__(self) -> None: ...
    def setBuilder(self, expr: _NlExprLike, sense: str, rhs: float) -> None: ...
    def getNlExpr(self) -> NlExpr: ...
    def getSense(self) -> str: ...

class NlConstrBuilderArray:
    def __init__(
        self,
        constrs: Optional[
            Union[
                NlConstrBuilder,
                Iterable[NlConstrBuilder],
                Mapping[Any, NlConstrBuilder],
            ]
        ] = None,
    ) -> None: ...
    def __getitem__(self, idx: int) -> NlConstrBuilder: ...
    def __iter__(self) -> NlConstrBuilderArray: ...
    def __next__(self) -> NlConstrBuilder: ...
    def pushBack(
        self,
        qconstrbuilder: Union[
            NlConstrBuilder, Iterable[NlConstrBuilder], Mapping[Any, NlConstrBuilder]
        ],
    ) -> None: ...
    def getBuilder(self, idx: int) -> NlConstrBuilder: ...
    def getSize(self) -> int: ...
    @property
    def size(self) -> int: ...

class QConstraint:
    name: str = ...
    rhs: float = ...
    sense: str = ...
    def getName(self) -> str: ...
    def getIdx(self) -> int: ...
    def setName(self, newname: str) -> None: ...
    def getRhs(self) -> float: ...
    def getSense(self) -> str: ...
    def setRhs(self, rhs: float) -> None: ...
    def setSense(self, sense: str) -> None: ...
    def getInfo(self, infoname: str) -> float: ...
    def setInfo(self, infoname: str, newval: float) -> None: ...
    def remove(self) -> None: ...

class QConstrArray:
    def __init__(
        self,
        constrs: Optional[
            Union[QConstraint, Iterable[QConstraint], Mapping[Any, QConstraint]]
        ] = None,
    ) -> None: ...
    def __getitem__(self, idx: int) -> QConstraint: ...
    def __iter__(self) -> QConstrArray: ...
    def __next__(self) -> QConstraint: ...
    def pushBack(
        self,
        qconstrs: Union[QConstraint, Iterable[QConstraint], Mapping[Any, QConstraint]],
    ) -> None: ...
    def getQConstr(self, idx: int) -> QConstraint: ...
    def getSize(self) -> int: ...
    @property
    def size(self) -> int: ...

class QConstrBuilder:
    def __init__(self) -> None: ...
    def setBuilder(self, expr: _QuadExprLike, sense: str, rhs: float) -> None: ...
    def getQuadExpr(self) -> QuadExpr: ...
    def getSense(self) -> str: ...

class QConstrBuilderArray:
    def __init__(
        self,
        constrs: Optional[
            Union[
                QConstrBuilder, Iterable[QConstrBuilder], Mapping[Any, QConstrBuilder]
            ]
        ] = None,
    ) -> None: ...
    def __getitem__(self, idx: int) -> QConstrBuilder: ...
    def __iter__(self) -> QConstrBuilderArray: ...
    def __next__(self) -> QConstrBuilder: ...
    def pushBack(
        self,
        qconstrbuilder: Union[
            QConstrBuilder, Iterable[QConstrBuilder], Mapping[Any, QConstrBuilder]
        ],
    ) -> None: ...
    def getBuilder(self, idx: int) -> QConstrBuilder: ...
    def getSize(self) -> int: ...
    @property
    def size(self) -> int: ...

class PsdConstraint:
    # mutable info or params
    name: str = ...

    def getName(self) -> str: ...
    def getIdx(self) -> int: ...
    def setName(self, newname: str) -> None: ...
    def getInfo(self, infoname: str) -> float: ...
    def setInfo(self, infoname: str, newval: float) -> None: ...
    def remove(self) -> None: ...

    # read-only info or attributes
    @property
    def index(self) -> int: ...
    @property
    def slack(self) -> float: ...
    @property
    def dual(self) -> float: ...

class PsdConstrArray:
    def __init__(
        self,
        constrs: Optional[
            Union[PsdConstraint, Iterable[PsdConstraint], Mapping[Any, PsdConstraint]]
        ] = None,
    ) -> None: ...
    def __getitem__(self, idx: int) -> PsdConstraint: ...
    def __iter__(self) -> PsdConstrArray: ...
    def __next__(self) -> PsdConstraint: ...
    def pushBack(
        self,
        constrs: Union[
            PsdConstraint, Iterable[PsdConstraint], Mapping[Any, PsdConstraint]
        ],
    ) -> None: ...
    def getPsdConstr(self, idx: int) -> PsdConstraint: ...
    def getSize(self) -> int: ...
    @property
    def size(self) -> int: ...

class PsdConstrBuilder:
    def __init__(self) -> None: ...
    def setBuilder(self, expr: _PsdExprLike, sense: str, rhs: float) -> None: ...
    def setRange(self, expr: _PsdExprLike, range: float) -> None: ...
    def getPsdExpr(self) -> PsdExpr: ...
    def getSense(self) -> str: ...
    def getRange(self) -> float: ...

class PsdConstrBuilderArray:
    def __init__(
        self,
        constrs: Optional[
            Union[
                PsdConstrBuilder,
                Iterable[PsdConstrBuilder],
                Mapping[Any, PsdConstrBuilder],
            ]
        ] = None,
    ) -> None: ...
    def __getitem__(self, idx: int) -> PsdConstrBuilder: ...
    def __iter__(self) -> PsdConstrBuilderArray: ...
    def __next__(self) -> PsdConstrBuilder: ...
    def pushBack(
        self,
        builder: Union[
            PsdConstrBuilder, Iterable[PsdConstrBuilder], Mapping[Any, PsdConstrBuilder]
        ],
    ) -> None: ...
    def getBuilder(self, idx: int) -> PsdConstrBuilder: ...
    def getSize(self) -> int: ...
    @property
    def size(self) -> int: ...

class LmiConstraint:
    # mutable info or params
    name: str = ...

    def getDim(self) -> int: ...
    def getLen(self) -> int: ...
    def getIdx(self) -> int: ...
    def getName(self) -> str: ...
    def setName(self, newname: str) -> None: ...
    def getInfo(self, infoname: str) -> float: ...
    def setRhs(self) -> SymMatrix: ...
    def remove(self) -> None: ...

    # read-only info or attributes
    @property
    def slack(self) -> List[float]: ...
    @property
    def dual(self) -> List[float]: ...
    @property
    def index(self) -> int: ...
    @property
    def dim(self) -> int: ...
    @property
    def len(self) -> int: ...
    @property
    def size(self) -> Tuple[int, int]: ...
    @property
    def shape(self) -> Tuple[int, int]: ...

class LmiConstrArray:
    def __init__(
        self,
        constrs: Optional[
            Union[LmiConstraint, Iterable[LmiConstraint], Mapping[Any, LmiConstraint]]
        ] = None,
    ) -> None: ...
    def __getitem__(self, idx: int) -> LmiConstraint: ...
    def __iter__(self) -> LmiConstrArray: ...
    def __next__(self) -> LmiConstraint: ...
    def pushBack(
        self,
        constrs: Union[
            LmiConstraint, Iterable[LmiConstraint], Mapping[Any, LmiConstraint]
        ],
    ) -> None: ...
    def getLmiConstr(self, idx: int) -> LmiConstraint: ...
    def getSize(self) -> int: ...
    def reserve(self, n) -> None: ...
    @property
    def size(self) -> int: ...

class Column:
    @overload
    def __init__(
        self, constrs: Union[Constraint, Column], coeffs: Optional[float] = None
    ) -> None: ...
    @overload
    def __init__(
        self,
        constrs: Union[ConstrArray, Iterable[Constraint]],
        coeffs: Union[float, Sequence[float]],
    ) -> None: ...
    @overload
    def __init__(
        self, constrs: Mapping[_T, Constraint], coeffs: Union[float, Mapping[_T, float]]
    ) -> None: ...
    @overload
    def __init__(
        self,
        constrs: Iterable[tuple[Constraint, float]],
    ) -> None: ...
    @overload
    def __init__(self) -> None: ...
    def setCoeff(self, idx: int, newval: float) -> None: ...
    def getCoeff(self, idx: int) -> float: ...
    def getConstr(self, idx: int) -> Constraint: ...
    def getSize(self) -> int: ...
    def addTerm(self, constr: Constraint, coeff: float = 1.0) -> None: ...
    @overload
    def addTerms(
        self,
        constrs: Union[Constraint, ConstrArray, Iterable[Constraint]],
        coeffs: Union[float, Sequence[float]],
    ) -> None: ...
    @overload
    def addTerms(
        self, constrs: Mapping[_T, Constraint], coeffs: Union[float, Mapping[_T, float]]
    ) -> None: ...
    def addColumn(self, col: Column, mult: float = 1.0) -> None: ...
    def clone(self) -> LinExpr: ...
    def remove(self, item: Union[int, Constraint]) -> None: ...
    def clear(self) -> None: ...
    @property
    def size(self) -> int: ...

class ColumnArray:
    def __init__(
        self,
        constrs: Optional[Union[Column, Iterable[Column], Mapping[Any, Column]]] = None,
    ) -> None: ...
    def __getitem__(self, idx: int) -> Column: ...
    def __iter__(self) -> ColumnArray: ...
    def __next__(self) -> Column: ...
    def pushBack(
        self, column: Union[Column, Iterable[Column], Mapping[Any, Column]]
    ) -> None: ...
    def getColumn(self, idx: int) -> Column: ...
    def getSize(self) -> int: ...
    def clear(self) -> None: ...
    @property
    def size(self) -> int: ...

class ExprBuilder:
    @overload
    def __init__(
        self,
        arg1: Union[Var, ExprBuilder, Iterable[Var]],
        arg2: Optional[Union[_Scalar, Iterable[_Scalar]]] = None,
    ) -> None: ...
    @overload
    def __init__(self, arg1: _Scalar = 0.0) -> None: ...
    def getSize(self) -> int: ...
    def getCoeff(self, idx: int) -> float: ...
    def getVar(self, idx: int) -> Var: ...
    def getConstant(self) -> float: ...
    def getValue(self) -> float: ...
    def addTerm(self, var: Var, coeff: float = 1.0) -> None: ...
    def addExpr(self, expr: ExprBuilder, mult: float = 1.0) -> None: ...
    def clone(self) -> ExprBuilder: ...
    def getExpr(self) -> LinExpr: ...
    @property
    def size(self) -> int: ...
    @overload
    def __add__(self, other: Union[float, Var, ExprBuilder]) -> ExprBuilder: ...
    @overload
    def __add__(self, other: LinExpr) -> LinExpr: ...
    @overload
    def __add__(self, other: QuadExpr) -> QuadExpr: ...
    @overload
    def __add__(self, other: PsdExpr) -> PsdExpr: ...
    @overload
    def __radd__(self, other: Union[float, Var, ExprBuilder]) -> ExprBuilder: ...
    @overload
    def __radd__(self, other: LinExpr) -> LinExpr: ...
    @overload
    def __radd__(self, other: QuadExpr) -> QuadExpr: ...
    @overload
    def __radd__(self, other: PsdExpr) -> PsdExpr: ...
    def __iadd__(self, other: Union[float, Var, ExprBuilder]) -> ExprBuilder: ...
    @overload
    def __sub__(self, other: Union[float, Var, ExprBuilder]) -> ExprBuilder: ...
    @overload
    def __sub__(self, other: LinExpr) -> LinExpr: ...
    @overload
    def __sub__(self, other: QuadExpr) -> QuadExpr: ...
    @overload
    def __sub__(self, other: PsdExpr) -> PsdExpr: ...
    @overload
    def __rsub__(self, other: Union[float, Var, ExprBuilder]) -> ExprBuilder: ...
    @overload
    def __rsub__(self, other: LinExpr) -> LinExpr: ...
    @overload
    def __rsub__(self, other: QuadExpr) -> QuadExpr: ...
    @overload
    def __rsub__(self, other: PsdExpr) -> PsdExpr: ...
    def __isub__(self, other: Union[float, Var, ExprBuilder]) -> ExprBuilder: ...
    @overload
    def __mul__(self, other: float) -> ExprBuilder: ...
    @overload
    def __mul__(self, other: Union[Var, ExprBuilder, LinExpr]) -> QuadExpr: ...
    @overload
    def __rmul__(self, other: float) -> ExprBuilder: ...
    @overload
    def __rmul__(self, other: Union[Var, ExprBuilder, LinExpr]) -> QuadExpr: ...
    def __imul__(self, other: float) -> ExprBuilder: ...
    def __pow__(self, other: int) -> QuadExpr: ...
    def __truediv__(self, other: float) -> ExprBuilder: ...
    def __div__(self, other: float) -> ExprBuilder: ...
    def __neg__(self) -> ExprBuilder: ...
    # '==', '<=' and '>=' are used as constraint sense
    @overload
    def __eq__(self, other: _LinExprLike) -> ConstrBuilder: ...
    @overload
    def __eq__(self, other: QuadExpr) -> QConstrBuilder: ...
    @overload
    def __eq__(self, other: PsdExpr) -> PsdConstrBuilder: ...
    @overload
    def __le__(self, other: _LinExprLike) -> ConstrBuilder: ...
    @overload
    def __le__(self, other: QuadExpr) -> QConstrBuilder: ...
    @overload
    def __le__(self, other: PsdExpr) -> PsdConstrBuilder: ...
    @overload
    def __ge__(self, other: _LinExprLike) -> ConstrBuilder: ...
    @overload
    def __ge__(self, other: QuadExpr) -> QConstrBuilder: ...
    @overload
    def __ge__(self, other: PsdExpr) -> PsdConstrBuilder: ...

class LinExpr:
    # mutable info or params
    constant: float = ...

    @overload
    def __init__(
        self, arg1: Union[Var, ExprBuilder, LinExpr], arg2: Optional[float] = None
    ) -> None: ...
    @overload
    def __init__(
        self, arg1: Union[VarArray, Iterable[Var]], arg2: Union[float, Sequence[float]]
    ) -> None: ...
    @overload
    def __init__(
        self, arg1: Mapping[_T, Var], arg2: Union[float, Mapping[_T, float]]
    ) -> None: ...
    @overload
    def __init__(
        self,
        arg1: Iterable[tuple[Var, float]],
    ) -> None: ...
    @overload
    def __init__(self, arg1: _Scalar = 0.0) -> None: ...
    def setCoeff(self, idx: int, newval: float) -> None: ...
    def getCoeff(self, idx: int) -> float: ...
    def getVar(self, idx: int) -> Var: ...
    def getConstant(self) -> float: ...
    def getValue(self) -> float: ...
    def getSize(self) -> int: ...
    def setConstant(self, newval: float) -> None: ...
    def addConstant(self, newval: float) -> None: ...
    def addTerm(self, var: Var, coeff: float = 1.0) -> None: ...
    @overload
    def addTerms(
        self,
        vars: Union[Var, VarArray, Iterable[Var]],
        coeffs: Union[float, Sequence[float]],
    ) -> None: ...
    @overload
    def addTerms(
        self, vars: Mapping[_T, Var], coeffs: Union[float, Mapping[_T, float]]
    ) -> None: ...
    def addExpr(self, expr: Union[ExprBuilder, LinExpr], mult: float = 1.0) -> None: ...
    def clone(self) -> LinExpr: ...
    def reserve(self, n: int) -> None: ...
    def remove(self, item: Union[int, Var]) -> None: ...

    # read-only info or attributes
    @property
    def value(self) -> float: ...
    @property
    def size(self) -> int: ...
    @overload
    def __add__(self, other: Union[float, Var, ExprBuilder, LinExpr]) -> LinExpr: ...
    @overload
    def __add__(self, other: QuadExpr) -> QuadExpr: ...
    @overload
    def __add__(self, other: PsdExpr) -> PsdExpr: ...
    @overload
    def __add__(self, other: Union[MVar, MLinExpr]) -> MLinExpr: ...
    @overload
    def __add__(self, other: MQuadExpr) -> MQuadExpr: ...
    @overload
    def __radd__(self, other: Union[float, Var, ExprBuilder, LinExpr]) -> LinExpr: ...
    @overload
    def __radd__(self, other: QuadExpr) -> QuadExpr: ...
    @overload
    def __radd__(self, other: PsdExpr) -> PsdExpr: ...
    def __iadd__(self, other: Union[float, Var, ExprBuilder, LinExpr]) -> LinExpr: ...
    @overload
    def __sub__(self, other: Union[float, Var, ExprBuilder, LinExpr]) -> LinExpr: ...
    @overload
    def __sub__(self, other: QuadExpr) -> QuadExpr: ...
    @overload
    def __sub__(self, other: PsdExpr) -> PsdExpr: ...
    @overload
    def __sub__(self, other: Union[MVar, MLinExpr]) -> MLinExpr: ...
    @overload
    def __sub__(self, other: MQuadExpr) -> MQuadExpr: ...
    @overload
    def __rsub__(self, other: Union[float, Var, ExprBuilder, LinExpr]) -> LinExpr: ...
    @overload
    def __rsub__(self, other: QuadExpr) -> QuadExpr: ...
    @overload
    def __rsub__(self, other: PsdExpr) -> PsdExpr: ...
    def __isub__(self, other: Union[float, Var, ExprBuilder, LinExpr]) -> LinExpr: ...
    @overload
    def __mul__(self, other: float) -> LinExpr: ...
    @overload
    def __mul__(self, other: Union[Var, ExprBuilder, LinExpr]) -> QuadExpr: ...
    @overload
    def __mul__(self, other: Union[MVar, MLinExpr]) -> MQuadExpr: ...
    @overload
    def __rmul__(self, other: float) -> LinExpr: ...
    @overload
    def __rmul__(self, other: Union[Var, ExprBuilder, LinExpr]) -> QuadExpr: ...
    def __imul__(self, other: float) -> LinExpr: ...
    def __pow__(self, other: int) -> QuadExpr: ...
    def __truediv__(self, other: float) -> LinExpr: ...
    def __itruediv__(self, other: float) -> LinExpr: ...
    def __neg__(self) -> LinExpr: ...
    # '==', '<=' and '>=' are used as constraint sense
    @overload
    def __eq__(self, other: _LinExprLike) -> ConstrBuilder: ...
    @overload
    def __eq__(self, other: QuadExpr) -> QConstrBuilder: ...
    @overload
    def __eq__(self, other: PsdExpr) -> PsdConstrBuilder: ...
    @overload
    def __eq__(self, other: Union[MVar, MLinExpr]) -> MConstrBuilder: ...
    @overload
    def __eq__(self, other: MQuadExpr) -> MQConstrBuilder: ...
    @overload
    def __le__(self, other: _LinExprLike) -> ConstrBuilder: ...
    @overload
    def __le__(self, other: QuadExpr) -> QConstrBuilder: ...
    @overload
    def __le__(self, other: PsdExpr) -> PsdConstrBuilder: ...
    @overload
    def __le__(self, other: Union[MVar, MLinExpr]) -> MConstrBuilder: ...
    @overload
    def __le__(self, other: MQuadExpr) -> MQConstrBuilder: ...
    @overload
    def __ge__(self, other: _LinExprLike) -> ConstrBuilder: ...
    @overload
    def __ge__(self, other: QuadExpr) -> QConstrBuilder: ...
    @overload
    def __ge__(self, other: PsdExpr) -> PsdConstrBuilder: ...
    @overload
    def __ge__(self, other: Union[MVar, MLinExpr]) -> MConstrBuilder: ...
    @overload
    def __ge__(self, other: MQuadExpr) -> MQConstrBuilder: ...

class NlExpr:
    constant: float = ...

    @overload
    def __init__(
        self, arg1: Union[Var, ExprBuilder, LinExpr, QuadExpr], arg2: Optional[float] = None
    ) -> None: ...
    @overload
    def __init__(self, arg1: _Scalar = 0.0) -> None: ...

    def getConstant(self) -> float: ...
    def getSize(self) -> int: ...
    def setConstant(self, newval: float) -> None: ...
    def addConstant(self, newval: float) -> None: ...
    def addTerm(self, var: Var, coeff: float = 1.0) -> None: ...
    @overload
    def addTerms(
        self,
        vars: Union[Var, VarArray, Iterable[Var]],
        coeffs: Union[float, Sequence[float]],
    ) -> None: ...
    @overload
    def addTerms(
        self, vars: Mapping[_T, Var], coeffs: Union[float, Mapping[_T, float]]
    ) -> None: ...
    def addLinExpr(
        self, expr: Union[ExprBuilder, LinExpr], mult: float = 1.0
    ) -> None: ...
    def addQuadExpr(self, expr: QuadExpr, mult: float = 1.0) -> None: ...
    def addNlExpr(self, expr: NlExpr, mult: float = 1.0) -> None: ...
    def clear(self) -> None: ...
    def clone(self) -> LinExpr: ...
    def getValue(self) -> float: ...
    def negate(self) -> None: ...
    def reserve(self, n: int) -> None: ...

    # read-only info or attributes
    @property
    def size(self) -> int: ...

class QuadExpr:
    # mutable info or params
    constant: float = ...

    @overload
    def __init__(
        self, arg1: Union[Var, ExprBuilder, LinExpr], arg2: Optional[float] = None
    ) -> None: ...
    @overload
    def __init__(
        self,
        arg1: Union[ExprBuilder, LinExpr],
        arg2: Union[Var, ExprBuilder, LinExpr] = None,
    ) -> None: ...
    @overload
    def __init__(self, arg1: QuadExpr, arg2: Optional[float] = None) -> None: ...
    @overload
    def __init__(self, arg1: float = 0.0) -> None: ...
    def setCoeff(self, idx: int, newval: float) -> None: ...
    def getCoeff(self, idx: int) -> float: ...
    def getVar1(self, idx: int) -> Var: ...
    def getVar2(self, idx: int) -> Var: ...
    def getLinExpr(self) -> LinExpr: ...
    def getConstant(self) -> float: ...
    def getValue(self) -> float: ...
    def getSize(self) -> int: ...
    def setConstant(self, newval: float) -> None: ...
    def addConstant(self, newval: float) -> None: ...
    def addTerm(self, coeff: float, var1: Var, var2: Var = None) -> None: ...
    def addTerms(
        self,
        coeffs: Union[float, Sequence[float]],
        vars1: Union[Var, VarArray, Iterable[Var], Mapping[Any, Var]],
        vars2: Optional[Union[Var, VarArray, Iterable[Var], Mapping[Any, Var]]] = None,
    ) -> None: ...
    def addLinExpr(
        self, expr: Union[ExprBuilder, LinExpr], mult: float = 1.0
    ) -> None: ...
    def addQuadExpr(self, expr: QuadExpr, mult: float = 1.0) -> None: ...
    def clone(self) -> QuadExpr: ...
    def reserve(self, n: int) -> None: ...
    def remove(self, item: Union[int, Var]) -> None: ...

    # read-only info or attributes
    @property
    def value(self) -> float: ...
    @property
    def size(self) -> int: ...
    @overload
    def __add__(self, other: _QuadExprLike) -> QuadExpr: ...
    @overload
    def __add__(self, other: Union[MVar, MLinExpr, MQuadExpr]) -> MQuadExpr: ...
    def __radd__(self, other: _QuadExprLike) -> QuadExpr: ...
    def __iadd__(self, other: _QuadExprLike) -> QuadExpr: ...
    @overload
    def __sub__(self, other: _QuadExprLike) -> QuadExpr: ...
    @overload
    def __sub__(self, other: Union[MVar, MLinExpr, MQuadExpr]) -> MQuadExpr: ...
    def __rsub__(self, other: _QuadExprLike) -> QuadExpr: ...
    def __isub__(self, other: _QuadExprLike) -> QuadExpr: ...
    def __mul__(self, other: float) -> QuadExpr: ...
    def __rmul__(self, other: float) -> QuadExpr: ...
    def __imul__(self, other: float) -> QuadExpr: ...
    def __truediv__(self, other: float) -> QuadExpr: ...
    def __itruediv__(self, other: float) -> QuadExpr: ...
    def __neg__(self) -> QuadExpr: ...
    # '==', '<=' and '>=' are used as constraint sense
    @overload
    def __eq__(self, other: _QuadExprLike) -> QConstrBuilder: ...
    @overload
    def __eq__(self, other: MQuadExpr) -> MQConstrBuilder: ...
    @overload
    def __le__(self, other: _QuadExprLike) -> QConstrBuilder: ...
    @overload
    def __le__(self, other: MQuadExpr) -> MQConstrBuilder: ...
    @overload
    def __ge__(self, other: _QuadExprLike) -> QConstrBuilder: ...
    @overload
    def __ge__(self, other: MQuadExpr) -> MQConstrBuilder: ...

class PsdExpr:
    # mutable info or params
    constant: float = ...

    def __init__(self, expr: _PsdExprLike = 0.0) -> None: ...
    def setCoeff(self, idx: int, mat: SymMatrix) -> None: ...
    def getCoeff(self, idx: int) -> SymMatrix: ...
    def getPsdVar(self, idx: int) -> PsdVar: ...
    def getLinExpr(self) -> LinExpr: ...
    def getConstant(self) -> float: ...
    def getValue(self) -> float: ...
    def getSize(self) -> int: ...
    def setConstant(self, newval: float) -> None: ...
    def addConstant(self, newval: float) -> None: ...
    @overload
    def addTerm(self, var: PsdVar, mat: SymMatrix) -> None: ...
    @overload
    def addTerm(self, var: Var, mat: float) -> None: ...
    def addTerms(
        self,
        vars: Union[Var, VarArray, Iterable[Var], Mapping[Any, Var]],
        mats: Union[
            SymMatrix, SymMatrixArray, Iterable[SymMatrix], Mapping[Any, SymMatrix]
        ],
    ) -> None: ...
    def addLinExpr(
        self, expr: Union[ExprBuilder, LinExpr], mult: float = 1.0
    ) -> None: ...
    def addPsdExpr(self, expr: PsdExpr, mult: float = 1.0) -> None: ...
    def clone(self) -> PsdExpr: ...
    def reserve(self, n: int) -> None: ...
    def remove(self, item: Union[int, Var, PsdVar]) -> None: ...

    # read-only info or attributes
    @property
    def value(self) -> float: ...
    @property
    def size(self) -> int: ...
    def __add__(self, other: _PsdExprLike) -> PsdExpr: ...
    def __radd__(self, other: _PsdExprLike) -> PsdExpr: ...
    def __iadd__(self, other: _PsdExprLike) -> PsdExpr: ...
    def __sub__(self, other: _PsdExprLike) -> PsdExpr: ...
    def __rsub__(self, other: _PsdExprLike) -> PsdExpr: ...
    def __isub__(self, other: _PsdExprLike) -> PsdExpr: ...
    def __mul__(self, other: float) -> PsdExpr: ...
    def __rmul__(self, other: float) -> PsdExpr: ...
    def __imul__(self, other: float) -> PsdExpr: ...
    def __truediv__(self, other: float) -> PsdExpr: ...
    def __itruediv__(self, other: float) -> PsdExpr: ...
    def __neg__(self) -> PsdExpr: ...
    # '==', '<=' and '>=' are used as constraint sense
    def __eq__(self, other: _PsdExprLike) -> PsdConstrBuilder: ...
    def __le__(self, other: _PsdExprLike) -> PsdConstrBuilder: ...
    def __ge__(self, other: _PsdExprLike) -> PsdConstrBuilder: ...

class LmiExpr:
    # mutable info or params
    constant: SymMatExpr = ...

    @overload
    def __init__(
        self,
        arg1: Optional[Union[SymMatrix, SymMatExpr]] = None,
    ) -> None: ...
    @overload
    def __init__(self, arg1: Var, arg2: Union[SymMatrix, SymMatExpr]) -> None: ...
    @overload
    def __init__(
        self,
        arg1: Iterable[Var],
        arg2: Union[Iterable[SymMatrix], Iterable[SymMatExpr]],
    ) -> None: ...
    def setCoeff(self, idx: int, mat: SymMatrix) -> None: ...
    def getCoeff(self, idx: int) -> SymMatExpr: ...
    def getVar(self, idx: int) -> Var: ...
    def getConstant(self) -> SymMatExpr: ...
    def getSize(self) -> int: ...
    def setConstant(self, mat: SymMatrix) -> None: ...
    def addConstant(self, mat: SymMatExpr) -> None: ...
    def addTerm(self, var: Var, mat: Union[SymMatrix, SymMatExpr]) -> None: ...
    def addTerms(
        self,
        vars: Union[Var, VarArray, Iterable[Var], Mapping[Any, Var]],
        mats: Union[
            SymMatrix, SymMatrixArray, Iterable[SymMatrix], Mapping[Any, SymMatrix]
        ],
    ) -> None: ...
    def addLmiExpr(self, expr: LmiExpr, mult: float = 1.0) -> None: ...
    def clone(self) -> LmiExpr: ...
    def reserve(self, n: int) -> None: ...
    def remove(self, item: Union[int, Var]) -> None: ...

    # read-only info or attributes
    @property
    def size(self) -> int: ...
    def __add__(self, other: Union[LmiExpr, SymMatrix, SymMatExpr]) -> LmiExpr: ...
    def __radd__(self, other: Union[LmiExpr, SymMatrix, SymMatExpr]) -> LmiExpr: ...
    def __iadd__(self, other: Union[LmiExpr, SymMatrix, SymMatExpr]) -> LmiExpr: ...
    def __sub__(self, other: Union[LmiExpr, SymMatrix, SymMatExpr]) -> LmiExpr: ...
    def __rsub__(self, other: Union[LmiExpr, SymMatrix, SymMatExpr]) -> LmiExpr: ...
    def __isub__(self, other: Union[LmiExpr, SymMatrix, SymMatExpr]) -> LmiExpr: ...
    def __mul__(self, other: float) -> LmiExpr: ...
    def __rmul__(self, other: float) -> LmiExpr: ...
    def __imul__(self, other: float) -> LmiExpr: ...
    def __truediv__(self, other: float) -> LmiExpr: ...
    def __itruediv__(self, other: float) -> LmiExpr: ...
    def __neg__(self) -> LmiExpr: ...

class ProbBuffer:
    def __init__(self, buff: int = 0) -> None: ...
    def getData(self) -> str: ...
    def getSize(self) -> int: ...
    def reSize(self, sz: int) -> None: ...

class EnvrConfig:
    def __init__(self) -> None: ...
    def set(self, name: str, value: str) -> None: ...

class Envr:
    def __init__(self, arg: Optional[Union[str, EnvrConfig]] = None) -> None: ...
    def createModel(self, name: str = "") -> Model: ...
    def close(self) -> None: ...

class Model:
    # mutable info or params
    # TODO: add more if needed
    objsense: int = ...
    ObjSense: int = ...
    objconst: float = ...
    ObjConst: float = ...

    # read-only info or attributes
    # TODO: add more if needed
    @property
    def cols(self) -> int: ...
    @property
    def Cols(self) -> int: ...
    @property
    def psdcols(self) -> int: ...
    @property
    def PsdCols(self) -> int: ...
    @property
    def rows(self) -> int: ...
    @property
    def Rows(self) -> int: ...
    @property
    def elems(self) -> int: ...
    @property
    def Elems(self) -> int: ...
    @property
    def qelems(self) -> int: ...
    @property
    def QElems(self) -> int: ...
    @property
    def psdelems(self) -> int: ...
    @property
    def PsdElems(self) -> int: ...
    @property
    def symmats(self) -> int: ...
    @property
    def SymMats(self) -> int: ...
    @property
    def bins(self) -> int: ...
    @property
    def Bins(self) -> int: ...
    @property
    def ints(self) -> int: ...
    @property
    def Ints(self) -> int: ...
    @property
    def soss(self) -> int: ...
    @property
    def Soss(self) -> int: ...
    @property
    def cones(self) -> int: ...
    @property
    def Cones(self) -> int: ...
    @property
    def expcones(self) -> int: ...
    @property
    def ExpCones(self) -> int: ...
    @property
    def qconstrs(self) -> int: ...
    @property
    def QConstrs(self) -> int: ...
    @property
    def psdconstrs(self) -> int: ...
    @property
    def PsdConstrs(self) -> int: ...
    @property
    def lmiconstrs(self) -> int: ...
    @property
    def LmiConstrs(self) -> int: ...
    @property
    def indicators(self) -> int: ...
    @property
    def Indicators(self) -> int: ...
    @property
    def iiscols(self) -> int: ...
    @property
    def IISCols(self) -> int: ...
    @property
    def iisrows(self) -> int: ...
    @property
    def IISRows(self) -> int: ...
    @property
    def iissoss(self) -> int: ...
    @property
    def IISSoss(self) -> int: ...
    @property
    def iisindicators(self) -> int: ...
    @property
    def IISIndicators(self) -> int: ...
    @property
    def objsense(self) -> int: ...
    @property
    def ObjSense(self) -> int: ...
    @property
    def status(self) -> int: ...
    @property
    def Status(self) -> int: ...
    @property
    def lpstatus(self) -> int: ...
    @property
    def LpStatus(self) -> int: ...
    @property
    def mipstatus(self) -> int: ...
    @property
    def MipStatus(self) -> int: ...
    @property
    def simplexiter(self) -> int: ...
    @property
    def SimplexIter(self) -> int: ...
    @property
    def barrieriter(self) -> int: ...
    @property
    def BarrierIter(self) -> int: ...
    @property
    def nodecnt(self) -> int: ...
    @property
    def NodeCnt(self) -> int: ...
    @property
    def poolsols(self) -> int: ...
    @property
    def PoolSols(self) -> int: ...
    @property
    def tuneresults(self) -> int: ...
    @property
    def TuneResults(self) -> int: ...
    @property
    def haslpsol(self) -> int: ...
    @property
    def HasLpSol(self) -> int: ...
    @property
    def hasdualfarkas(self) -> int: ...
    @property
    def HasDualFarkas(self) -> int: ...
    @property
    def hasprimalray(self) -> int: ...
    @property
    def HasPrimalRay(self) -> int: ...
    @property
    def hasbasis(self) -> int: ...
    @property
    def HasBasis(self) -> int: ...
    @property
    def hasmipsol(self) -> int: ...
    @property
    def HasMipSol(self) -> int: ...
    @property
    def hasqobj(self) -> int: ...
    @property
    def HasQObj(self) -> int: ...
    @property
    def haspsdobj(self) -> int: ...
    @property
    def HasPsdObj(self) -> int: ...
    @property
    def hasIIS(self) -> int: ...
    @property
    def HasIIS(self) -> int: ...
    @property
    def hasFeasRelaxSol(self) -> int: ...
    @property
    def HasFeasRelaxSol(self) -> int: ...
    @property
    def ismip(self) -> int: ...
    @property
    def IsMIP(self) -> int: ...
    @property
    def isminiis(self) -> int: ...
    @property
    def IsMinIIS(self) -> int: ...
    @property
    def solvingtime(self) -> float: ...
    @property
    def SolvingTime(self) -> float: ...
    @property
    def objconst(self) -> float: ...
    @property
    def ObjConst(self) -> float: ...
    @property
    def objval(self) -> float: ...
    @property
    def ObjVal(self) -> float: ...
    @property
    def lpobjval(self) -> float: ...
    @property
    def LpObjVal(self) -> float: ...
    @property
    def bestobj(self) -> float: ...
    @property
    def BestObj(self) -> float: ...
    @property
    def bestbnd(self) -> float: ...
    @property
    def BestBnd(self) -> float: ...
    @property
    def bestgap(self) -> float: ...
    @property
    def BestGap(self) -> float: ...
    @property
    def feasrelaxobj(self) -> float: ...
    @property
    def FeasRelaxObj(self) -> float: ...
    @property
    def param(self) -> ParamSetting: ...
    @property
    def Param(self) -> ParamSetting: ...
    def addVar(
        self,
        lb: float = 0.0,
        ub: float = COPT.INFINITY,
        obj: float = 0.0,
        vtype: str = COPT.CONTINUOUS,
        name: str = "",
        column: Optional[Column] = None,
    ) -> Var: ...

    # single int arg + scalar kwargs
    @overload
    def addVars(
        self,
        __indices: int,
        *,
        lb: float = 0.0,
        ub: float = COPT.INFINITY,
        obj: float = 0.0,
        vtype: str = COPT.CONTINUOUS,
        nameprefix: str = "C"
    ) -> tupledict[int, Var]: ...

    # multiple int args + scalar kwargs
    @overload
    def addVars(
        self,
        __indices1: int,
        __indices2: int,
        *indices: int,
        lb: float = 0.0,
        ub: float = COPT.INFINITY,
        obj: float = 0.0,
        vtype: str = COPT.CONTINUOUS,
        nameprefix: str = "C"
    ) -> tupledict[Tuple[int, ...], Var]: ...

    # single list of scalar or tuple arg + scalar/list kwargs
    @overload
    def addVars(
        self,
        __indices: Iterable[Union[_Scalar, Tuple[_Scalar, ...]]],
        *,
        lb: Union[float, Iterable[float], Iterable[Iterable[float]]] = 0.0,
        ub: Union[float, Iterable[float], Iterable[Iterable[float]]] = COPT.INFINITY,
        obj: Union[float, Iterable[float], Iterable[Iterable[float]]] = 0.0,
        vtype: Union[str, Iterable[str], Iterable[Iterable[str]]] = COPT.CONTINUOUS,
        nameprefix: str = "C"
    ) -> tupledict[Any, Var]: ...

    # single list of scalar or tuple arg + scalar/dict kwargs
    @overload
    def addVars(
        self,
        __indices: Iterable[Union[_Scalar, Tuple[_Scalar, ...]]],
        *,
        lb: Union[float, Mapping[Any, float]] = 0.0,
        ub: Union[float, Mapping[Any, float]] = COPT.INFINITY,
        obj: Union[float, Mapping[Any, float]] = 0.0,
        vtype: Union[str, Mapping[Any, str]] = COPT.CONTINUOUS,
        nameprefix: str = "C"
    ) -> tupledict[Any, Var]: ...

    # multiple list of scalar or tuple args + scalar/list kwargs
    @overload
    def addVars(
        self,
        __indices1: Iterable[Union[_Scalar, Tuple[_Scalar, ...]]],
        __indices2: Iterable[Union[_Scalar, Tuple[_Scalar, ...]]],
        *indices: Iterable[Union[_Scalar, Tuple[_Scalar, ...]]],
        lb: Union[float, Iterable[float], Iterable[Iterable[float]]] = 0.0,
        ub: Union[float, Iterable[float], Iterable[Iterable[float]]] = COPT.INFINITY,
        obj: Union[float, Iterable[float], Iterable[Iterable[float]]] = 0.0,
        vtype: Union[str, Iterable[str], Iterable[Iterable[str]]] = COPT.CONTINUOUS,
        nameprefix: str = "C"
    ) -> tupledict[Tuple[Any, ...], Var]: ...

    # multiple list of scalar or tuple args + scalar/dict kwargs
    @overload
    def addVars(
        self,
        __indices1: Iterable[Union[_Scalar, Tuple[_Scalar, ...]]],
        __indices2: Iterable[Union[_Scalar, Tuple[_Scalar, ...]]],
        *indices: Iterable[Union[_Scalar, Tuple[_Scalar, ...]]],
        lb: Union[float, Mapping[Any, float]] = 0.0,
        ub: Union[float, Mapping[Any, float]] = COPT.INFINITY,
        obj: Union[float, Mapping[Any, float]] = 0.0,
        vtype: Union[str, Mapping[Any, str]] = COPT.CONTINUOUS,
        nameprefix: str = "C"
    ) -> tupledict[Tuple[Any, ...], Var]: ...
    def addMVar(
        self,
        shape: Union[int, Tuple[int, ...]],
        lb: Union[float, Iterable[float], Iterable[Iterable[float]]] = 0.0,
        ub: Union[float, Iterable[float], Iterable[Iterable[float]]] = COPT.INFINITY,
        obj: Union[float, Iterable[float], Iterable[Iterable[float]]] = 0.0,
        vtype: Union[str, Iterable[str], Iterable[Iterable[str]]] = COPT.CONTINUOUS,
        nameprefix: str = "C",
    ) -> MVar: ...
    @overload
    def addConstr(
        self, lhs: _LinExprLike, sense: str, rhs: _LinExprLike, name: str = ""
    ) -> Constraint: ...
    @overload
    def addConstr(
        self, lhs: PsdExpr, sense: str, rhs: _PsdExprLike, name: str = ""
    ) -> PsdConstraint: ...
    @overload
    def addConstr(
        self, lhs: _PsdExprLike, sense: str, rhs: PsdExpr, name: str = ""
    ) -> PsdConstraint: ...
    @overload
    def addConstr(self, lhs: bool, name: str = "") -> Constraint: ...
    @overload
    def addConstr(self, lhs: ConstrBuilder, name: str = "") -> Constraint: ...
    @overload
    def addConstr(
        self, lhs: Tuple[ConstrBuilder, str, float], name: str = ""
    ) -> Constraint: ...
    @overload
    def addConstr(self, lhs: GenConstrBuilder, name: str = "") -> GenConstr: ...
    @overload
    def addConstr(self, lhs: PsdConstrBuilder, name: str = "") -> PsdConstraint: ...
    @overload
    def addConstr(
        self, lhs: Tuple[PsdConstrBuilder, str, float], name: str = ""
    ) -> PsdConstraint: ...
    @overload
    def addConstr(self, lhs: LmiExpr, name: str = "") -> LmiConstraint: ...
    @overload
    def addBoundConstr(
        self,
        expr: _LinExprLike,
        lb: float = -COPT.INFINITY,
        ub: float = COPT.INFINITY,
        name: str = "",
    ) -> Constraint: ...
    @overload
    def addBoundConstr(
        self,
        expr: PsdExpr,
        lb: float = -COPT.INFINITY,
        ub: float = COPT.INFINITY,
        name: str = "",
    ) -> PsdConstraint: ...
    @overload
    def addConstrs(self, generator: bool, nameprefix: str = "R") -> Constraint: ...
    @overload
    def addConstrs(self, generator: int, nameprefix: str = "R") -> ConstrArray: ...
    @overload
    def addConstrs(
        self, generator: Iterable[ConstrBuilder], nameprefix: str = "R"
    ) -> tupledict[Any, Constraint]: ...
    @overload
    def addConstrs(
        self, generator: Generator[ConstrBuilder, None, None], nameprefix: str = "R"
    ) -> tupledict[Any, Constraint]: ...
    @overload
    def addConstrs(
        self, generator: MConstrBuilder, nameprefix: str = "R"
    ) -> MConstr: ...
    def addMConstr(
        self,
        A: np.ndarray,
        x: Union[None, MVar, Iterable[Var], Iterable[Iterable[Var]]],
        sense: str,
        b: Union[float, Iterable[float], Iterable[Iterable[float]]],
        nameprefix: str = "",
    ) -> MConstr: ...
    @overload
    def addSOS(
        self,
        sostype: int,
        vars: Union[Var, VarArray, Iterable[Var]],
        weights: Optional[Iterable[float]] = None,
    ) -> SOS: ...
    @overload
    def addSOS(
        self,
        sostype: int,
        vars: Mapping[_T, Var],
        weights: Optional[Mapping[_T, float]] = None,
    ) -> SOS: ...
    @overload
    def addSOS(
        self, sostype: SOSBuilder, vars: Any = None, weights: Any = None
    ) -> SOS: ...
    @overload
    def addGenConstrIndicator(
        self,
        binvar: Var,
        binval: int,
        lhs: _LinExprLike,
        sense: str,
        rhs: float,
        type: int = COPT.INDICATOR_IF,
        name: str = "",
    ) -> GenConstr: ...
    @overload
    def addGenConstrIndicator(
        self,
        binvar: Var,
        binval: int,
        lhs: ConstrBuilder,
        type: int = COPT.INDICATOR_IF,
        name: str = "",
    ) -> GenConstr: ...
    def addGenConstrIndicators(
        self,
        builders: List[GenConstrBuilder],
        nameprefix: str = "",
    ) -> GenConstrArray: ...
    def addGenConstrMax(
        self, resvar: Var, vars: Iterable[Var], constant: float = None, name: str = ""
    ) -> GenConstrX: ...
    def addGenConstrMin(
        self, resvar: Var, vars: Iterable[Var], constant: float = None, name: str = ""
    ) -> GenConstrX: ...
    def addGenConstrAbs(
        self, resvar: Var, argvar: Var, name: str = ""
    ) -> GenConstrX: ...
    def addGenConstrAnd(
        self, resvar: Var, vars: Iterable[Var], name: str = ""
    ) -> GenConstrX: ...
    def addGenConstrOr(
        self, resvar: Var, vars: Iterable[Var], name: str = ""
    ) -> GenConstrX: ...
    def addGenConstrPWL(
        self,
        xvar: Var,
        yvar: Var,
        xpts: Iterable[float],
        ypts: Iterable[float],
        name: str = "",
    ) -> GenConstrX: ...
    def addConeByDim(
        self,
        dim: int,
        ctype: int,
        vtype: Optional[Sequence[str]],
        nameprefix: str = "ConV",
    ) -> Cone: ...
    @overload
    def addCone(
        self, vars: Union[Var, VarArray, Iterable[Var], Mapping[Any, Var]], ctype: int
    ) -> Cone: ...
    @overload
    def addCone(self, vars: ConeBuilder) -> Cone: ...
    def addCones(
        self, vars: Union[ConeBuilderArray, MVar], ctype: int = None
    ) -> ConeArray: ...
    def addExpConeByDim(
        self,
        dim: int,
        ctype: int,
        vtype: Optional[Sequence[str]],
        nameprefix: str = "ExpConV",
    ) -> ExpCone: ...
    @overload
    def addExpCone(
        self, vars: Union[VarArray, Iterable[Var], Mapping[Any, Var]], ctype: int
    ) -> ExpCone: ...
    @overload
    def addExpCone(self, vars: ExpConeBuilder) -> ExpCone: ...
    def addExpCones(
        self, vars: Union[ExpConeBuilderArray, MVar], ctype: int = None
    ) -> ExpConeArray: ...
    @overload
    def addAffineCone(
        self,
        exprs: Union[_PsdExprLike, Iterable[_PsdExprLike], Mapping[Any, _PsdExprLike]],
        ctype: int,
        name: str,
    ) -> AffineCone: ...
    @overload
    def addAffineCone(self, builder: AffineConeBuilder, name: str) -> AffineCone: ...
    def addAffineCones(
        self,
        exprs: Union[AffineConeBuilderArray, MLinExpr, MPsdExpr],
        ctype: int = None,
        nameprefix: str = "AffineConV",
    ) -> AffineConeArray: ...
    @overload
    def addNlConstr(
        self, lhs: _NlExprLike, sense: str, rhs: _NlExprLike, name: str = ""
    ) -> NlConstraint: ...
    @overload
    def addNlConstr(
        self, lhs: Union[bool, ConstrBuilder], name: str = ""
    ) -> Constraint: ...
    @overload
    def addNlConstr(self, lhs: QConstrBuilder, name: str = "") -> QConstraint: ...
    @overload
    def addNlConstr(self, lhs: NlConstrBuilder, name: str = "") -> NlConstraint: ...
    @overload
    def addNlConstrs(
        self, generator: Iterable[NlConstrBuilder], nameprefix: str = "NR"
    ) -> tupledict[Any, NlConstraint]: ...
    @overload
    def addNlConstrs(self, generator: bool, nameprefix: str = "NR") -> Constraint: ...
    @overload
    def addNlConstrs(
        self, generator: Union[int, ConstrBuilder], nameprefix: str = "NR"
    ) -> Constraint: ...
    @overload
    def addNlConstrs(
        self, generator: QConstrBuilder, name: str = "NR"
    ) -> QConstraint: ...
    @overload
    def addNlConstrs(
        self, generator: NlConstrBuilder, name: str = "NR"
    ) -> NlConstraint: ...
    @overload
    def addQConstr(
        self, lhs: _LinExprLike, sense: str, rhs: _LinExprLike, name: str = ""
    ) -> Constraint: ...
    @overload
    def addQConstr(
        self, lhs: QuadExpr, sense: str, rhs: _QuadExprLike, name: str = ""
    ) -> QConstraint: ...
    @overload
    def addQConstr(
        self, lhs: _QuadExprLike, sense: str, rhs: QuadExpr, name: str = ""
    ) -> QConstraint: ...
    @overload
    def addQConstr(
        self, lhs: MQuadExpr, sense: str, rhs: _QuadExprLike, name: str = ""
    ) -> QConstraint: ...
    @overload
    def addQConstr(self, lhs: bool, name: str = "") -> Constraint: ...
    @overload
    def addQConstr(self, lhs: ConstrBuilder, name: str = "") -> Constraint: ...
    @overload
    def addQConstr(
        self, lhs: Tuple[ConstrBuilder, str, float], name: str = ""
    ) -> Constraint: ...
    @overload
    def addQConstr(self, lhs: QConstrBuilder, name: str = "") -> QConstraint: ...
    @overload
    def addQConstr(
        self, lhs: Tuple[QConstrBuilder, str, float], name: str = ""
    ) -> QConstraint: ...
    @overload
    def addQConstr(self, lhs: MConstrBuilder, name: str = "") -> MConstr: ...
    @overload
    def addQConstr(self, lhs: MQConstrBuilder, name: str = "") -> QConstraint: ...
    def addMQConstr(
        self,
        Q: Union[None, np.ndarray],
        c: Union[None, Iterable[float]],
        sense: str,
        rhs: Union[float, Iterable[float]],
        xQ_L: Union[MVar, Iterable[Var]] = None,
        xQ_R: Union[MVar, Iterable[Var]] = None,
        xc: Union[MVar, Iterable[Var]] = None,
        name: str = "",
    ) -> QConstraint: ...
    def addPsdVar(self, dim: int, name: str = "") -> PsdVar: ...
    @overload
    def addPsdVars(self, dims: int, nameprefix: str = "PSDV") -> PsdVar: ...
    @overload
    def addPsdVars(
        self, dims: Sequence[int], nameprefix: str = "PSDV"
    ) -> PsdVarArray: ...
    @overload
    def addUserCut(
        self, lhs: _LinExprLike, sense: str, rhs: _LinExprLike, name: str = ""
    ) -> None: ...
    @overload
    def addUserCut(self, lhs: ConstrBuilder, name: str = "") -> None: ...
    @overload
    def addUserCut(self, lhs: ConstrBuilderArray, name: str = "") -> None: ...
    @overload
    def addUserCuts(
        self, generator: ConstrBuilderArray, nameprefix: str = "R"
    ) -> None: ...
    @overload
    def addUserCuts(
        self, generator: Iterable[ConstrBuilder], nameprefix: str = "R"
    ) -> None: ...
    @overload
    def addUserCuts(
        self, generator: Generator[ConstrBuilder, None, None], nameprefix: str = "R"
    ) -> None: ...
    @overload
    def addLazyConstr(
        self, lhs: _LinExprLike, sense: str, rhs: _LinExprLike, name: str = ""
    ) -> None: ...
    @overload
    def addLazyConstr(self, lhs: ConstrBuilder, name: str = "") -> None: ...
    @overload
    def addLazyConstr(self, lhs: ConstrBuilderArray, name: str = "") -> None: ...
    @overload
    def addLazyConstrs(
        self, generator: ConstrBuilderArray, nameprefix: str = "R"
    ) -> None: ...
    @overload
    def addLazyConstrs(
        self, generator: Iterable[ConstrBuilder], nameprefix: str = "R"
    ) -> None: ...
    @overload
    def addLazyConstrs(
        self, generator: Generator[ConstrBuilder, None, None], nameprefix: str = "R"
    ) -> None: ...
    @overload
    def addSparseMat(
        self, dim: int, rows: Sequence[Tuple[int, int, float]]
    ) -> SymMatrix: ...
    @overload
    def addSparseMat(
        self, dim: int, rows: Sequence[int], cols: Sequence[int], vals: Sequence[float]
    ) -> SymMatrix: ...
    def addDenseMat(
        self, dim: int, vals: Union[float, Sequence[float]]
    ) -> SymMatrix: ...
    def addDiagMat(
        self, dim: int, vals: Union[float, Sequence[float]], offset: int = None
    ) -> SymMatrix: ...
    def addOnesMat(self, dim: int) -> SymMatrix: ...
    def addEyeMat(self, dim: int) -> SymMatrix: ...
    def addSymMat(self, expr: SymMatExpr) -> SymMatrix: ...
    def getSymMat(self, idx: int) -> SymMatrix: ...
    @overload
    def setObjective(
        self, expr: Union[_NlExprLike, PsdExpr], sense: int = None
    ) -> None: ...
    @overload
    def setObjective(
        self, expr: Union[MLinExpr, MQuadExpr], sense: int = None
    ) -> None: ...
    def setMObjective(
        self,
        Q: Union[None, np.ndarray],
        c: Union[None, Iterable[float], Iterable[Iterable[float]]],
        constant: float,
        xQ_L: Union[MVar, Iterable[Var], Iterable[Iterable[Var]]] = None,
        xQ_R: Union[MVar, Iterable[Var], Iterable[Iterable[Var]]] = None,
        xc: Union[MVar, Iterable[Var], Iterable[Iterable[Var]]] = None,
        sense: str = None,
    ) -> None: ...
    def setObjSense(self, sense: int) -> None: ...
    def setObjConst(self, constant: float) -> None: ...
    def getObjective(self) -> Union[LinExpr, NlExpr, PsdExpr, QuadExpr]: ...
    def delNlObj(self) -> None: ...
    def delQuadObj(self) -> None: ...
    def delPsdObj(self) -> None: ...
    def setObjectiveN(
        self,
        idx: int,
        expr: Union[float, Var, LinExpr, MVar, MLinExpr],
        sense: int = 0,
        priority: float = 0.0,
        weight: float = 1.0,
        abstol: float = 1e-6,
        reltol: float = 0.0,
    ) -> None: ...
    def getObjectiveN(self, idx: int) -> LinExpr: ...
    def delObjN(self, idx: int) -> None: ...
    def setObjParamN(self, idx: int, paramname: str, newval: float) -> None: ...
    def setParamN(
        self, idx: int, paramname: str, newval: Union[int, float]
    ) -> None: ...
    def getObjParamN(self, idx: int, paramname: str) -> Union[int, float]: ...
    def getParamN(self, idx: int, paramname: str) -> Union[int, float]: ...
    def getAttrN(self, idx: int, attrname: str) -> Union[int, float]: ...
    def getPoolObjValN(self, idx: int, isol: int) -> float: ...
    def resetObjParamN(self, idx: int) -> None: ...
    def resetParamN(self, idx: int) -> None: ...
    def getCol(self, var: Var) -> Column: ...
    def getRow(self, constr: Constraint) -> LinExpr: ...
    def getNlRow(self, constr: NlConstraint) -> NlExpr: ...
    def getQuadRow(self, qconstr: QConstraint) -> QuadExpr: ...
    def getPsdRow(self, constr: PsdConstraint) -> PsdExpr: ...
    def getLmiRow(self, constr: LmiConstraint) -> LmiExpr: ...
    def getVar(self, idx: int) -> Var: ...
    def getVarByName(self, name: str) -> Var: ...
    def getVars(self) -> VarArray: ...
    def getConstr(self, idx: int) -> Constraint: ...
    def getConstrByName(self, name: str) -> Constraint: ...
    def getConstrs(self) -> ConstrArray: ...
    @overload
    def getConstrBuilders(
        self, constrs: Optional[Union[ConstrArray, Sequence[Constraint]]] = None
    ) -> ConstrBuilderArray: ...
    @overload
    def getConstrBuilders(self, constrs: Constraint) -> ConstrBuilder: ...
    @overload
    def getConstrBuilders(
        self, constrs: Mapping[_T, Constraint]
    ) -> tupledict[_T, ConstrBuilder]: ...
    def getSOSs(self) -> SOSArray: ...
    def getSOS(self, sos: SOS) -> SOSBuilder: ...
    @overload
    def getSOSBuilders(self, soss: SOS) -> SOSBuilder: ...
    @overload
    def getSOSBuilders(
        self, soss: Union[Iterable[SOS], Mapping[Any, SOS]] = None
    ) -> SOSBuilderArray: ...
    def getGenConstr(self, idx: int) -> GenConstr: ...
    def getGenConstrByName(self, name: str) -> GenConstr: ...
    def getGenConstrs(self) -> GenConstrArray: ...
    def getGenConstrIndicators(
        self, genconstrs: List[GenConstr]
    ) -> GenConstrBuilderArray: ...
    def getGenConstrIndicator(self, genconstr: GenConstr) -> GenConstrBuilder: ...
    def getCones(self) -> ConeArray: ...
    @overload
    def getConeBuilders(self, cones: Cone) -> ConeBuilder: ...
    @overload
    def getConeBuilders(
        self, cones: Union[Iterable[Cone], Mapping[Any, Cone]] = None
    ) -> ConeBuilderArray: ...
    def getExpCones(self) -> ExpConeArray: ...
    @overload
    def getExpConeBuilders(self, cones: ExpCone) -> ExpConeBuilder: ...
    @overload
    def getExpConeBuilders(
        self, cones: Union[Iterable[ExpCone], Mapping[Any, ExpCone]] = None
    ) -> ExpConeBuilderArray: ...
    def getNlConstr(self, idx: int) -> NlConstraint: ...
    def getNlConstrByName(self, name: str) -> NlConstraint: ...
    def getNlConstrs(self) -> NlConstrArray: ...
    @overload
    def getNlConstrBuilders(self, constrs: NlConstraint) -> ConstrBuilder: ...
    @overload
    def getNlConstrBuilders(
        self, qconstrs: Optional[Union[NlConstrArray, Sequence[NlConstraint]]] = None
    ) -> NlConstrBuilderArray: ...
    @overload
    def getNlConstrBuilders(
        self, qconstrs: Mapping[_T, NlConstraint]
    ) -> tupledict[_T, NlConstrBuilder]: ...
    def getQConstr(self, idx: int) -> QConstraint: ...
    def getQConstrByName(self, name: str) -> QConstraint: ...
    def getQConstrs(self) -> QConstrArray: ...
    @overload
    def getQConstrBuilders(self, qconstrs: QConstraint) -> QConstrBuilder: ...
    @overload
    def getQConstrBuilders(
        self, qconstrs: Optional[Union[QConstrArray, Sequence[QConstraint]]] = None
    ) -> QConstrBuilderArray: ...
    @overload
    def getQConstrBuilders(
        self, qconstrs: Mapping[_T, QConstraint]
    ) -> tupledict[_T, QConstrBuilder]: ...
    def getPsdVar(self, idx: int) -> PsdVar: ...
    def getPsdVarByName(self, name: str) -> PsdVar: ...
    def getPsdVars(self) -> PsdVarArray: ...
    def getPsdConstr(self, idx: int) -> PsdConstraint: ...
    def getPsdConstrByName(self, name: str) -> PsdConstraint: ...
    def getPsdConstrs(self) -> PsdConstrArray: ...
    @overload
    def getPsdConstrBuilders(self, constrs: PsdConstraint) -> PsdConstrBuilder: ...
    @overload
    def getPsdConstrBuilders(
        self, constrs: Optional[Union[PsdConstrArray, Sequence[PsdConstraint]]] = None
    ) -> PsdConstrBuilderArray: ...
    @overload
    def getPsdConstrBuilders(
        self, constrs: Mapping[_T, PsdConstraint]
    ) -> tupledict[_T, PsdConstrBuilder]: ...
    def getLmiConstr(self, idx: int) -> LmiConstraint: ...
    def getLmiConstrByName(self, name: str) -> LmiConstraint: ...
    def getLmiConstrs(self) -> LmiConstrArray: ...
    def getLmiRhs(self, constr: LmiConstraint) -> SymMatrix: ...
    def setLmiRhs(self, constr: LmiConstraint, mat: SymMatrix) -> None: ...
    @overload
    def getCoeff(self, constr: Union[Constraint, LmiConstraint], var: Var) -> float: ...
    @overload
    def getCoeff(self, constr: PsdConstraint, var: PsdVar) -> SymMatrix: ...
    @overload
    def setCoeff(self, constr: Constraint, var: Var, newval: float) -> None: ...
    @overload
    def setCoeff(self, constr: LmiConstraint, var: Var, newval: SymMatrix) -> None: ...
    @overload
    def setCoeff(
        self, constr: PsdConstraint, var: PsdVar, newval: SymMatrix
    ) -> None: ...
    @overload
    def setCoeffs(self, constrs: Constraint, vars: Var, newvals: float) -> None: ...
    @overload
    def setCoeffs(
        self,
        constrs: Iterable[Constraint],
        vars: Iterable[Var],
        newvals: Union[float, Iterable[float]],
    ) -> None: ...
    @overload
    def setCoeffs(
        self,
        constrs: Mapping[_T, Constraint],
        vars: Mapping[_T, Var],
        newvals: Union[float, Mapping[_T, float]],
    ) -> None: ...
    @overload
    def setCoeffs(
        self,
        constrs: ConstrArray,
        vars: VarArray,
        newvals: Union[float, Iterable[float]],
    ) -> None: ...
    def getA(self) -> sparse.csc_matrix: ...
    def loadMatrix(
        self,
        c: Optional[Sequence[float]],
        A: sparse.csc_matrix,
        lhs: Sequence[float],
        rhs: Sequence[float],
        lb: Optional[Sequence[float]],
        ub: Optional[Sequence[float]],
        vtype: Optional[Sequence[str]] = None,
    ) -> None: ...
    def loadQ(self, Q: sparse.isspmatrix_coo) -> None: ...
    def loadCone(
        self,
        ncone: int,
        types: Optional[Sequence[int]],
        dims: Optional[Sequence[int]],
        indices: Optional[Sequence[int]],
    ) -> None: ...
    def loadExpCone(
        self,
        ncone: int,
        types: Optional[Sequence[int]],
        indices: Optional[Sequence[int]],
    ) -> None: ...
    def loadConeMatrix(
        self,
        c: Optional[Sequence[float]],
        A: sparse.isspmatrix_csc,
        b: Optional[Sequence[float]],
        K: Mapping[str, int],
        Q: Optional[sparse.isspmatrix_coo] = None,
        vtype: Sequence[str] = None,
    ) -> None: ...
    def getLpSolution(
        self,
    ) -> Tuple[List[float], List[float], List[float], List[float]]: ...
    def setLpSolution(
        self,
        values: Sequence[float],
        slack: Sequence[float],
        duals: Sequence[float],
        redcost: Sequence[float],
    ) -> None: ...
    def getPsdSolution(
        self,
    ) -> Tuple[List[float], List[float], List[float], List[float]]: ...
    def getLmiSolution(self) -> Tuple[List[float], List[float]]: ...
    def getValues(self) -> List[float]: ...
    def getRedcosts(self) -> List[float]: ...
    def getSlacks(self) -> List[float]: ...
    def getDuals(self) -> List[float]: ...
    def getPsdValues(self) -> List[float]: ...
    def getPsdRedcosts(self) -> List[float]: ...
    def getPsdSlacks(self) -> List[float]: ...
    def getPsdDuals(self) -> List[float]: ...
    def getLmiSlacks(self) -> List[float]: ...
    def getLmiDuals(self) -> List[float]: ...
    @overload
    def getVarBasis(self, vars: Var) -> int: ...
    @overload
    def getVarBasis(self, vars: Optional[Sequence[Var]] = None) -> List[int]: ...
    @overload
    def getVarBasis(self, vars: Mapping[_T, Var]) -> tupledict[_T, int]: ...
    @overload
    def getVarBasis(self, vars: MVar) -> np.ndarray: ...
    @overload
    def getConstrBasis(self, constrs: Constraint) -> int: ...
    @overload
    def getConstrBasis(
        self, constrs: Optional[Sequence[Constraint]] = None
    ) -> List[int]: ...
    @overload
    def getConstrBasis(
        self, constrs: Mapping[_T, Constraint]
    ) -> tupledict[_T, int]: ...
    @overload
    def getConstrBasis(self, constrs: MConstr) -> np.ndarray: ...
    def getPoolObjVal(self, isol: int) -> float: ...
    @overload
    def getPoolSolution(self, isol: int, vars: Var) -> float: ...
    @overload
    def getPoolSolution(
        self, isol: int, vars: Union[VarArray, Iterable[Var]]
    ) -> List[float]: ...
    @overload
    def getPoolSolution(
        self, isol: int, vars: Mapping[_T, Var]
    ) -> tupledict[_T, float]: ...
    @overload
    def getVarLowerIIS(self, vars: Var) -> int: ...
    @overload
    def getVarLowerIIS(self, vars: Union[VarArray, Iterable[Var]]) -> List[int]: ...
    @overload
    def getVarLowerIIS(self, vars: Mapping[_T, Var]) -> tupledict[_T, int]: ...
    @overload
    def getVarLowerIIS(self, vars: MVar) -> np.ndarray: ...
    @overload
    def getVarUpperIIS(self, vars: Var) -> int: ...
    @overload
    def getVarUpperIIS(self, vars: Union[VarArray, Iterable[Var]]) -> List[int]: ...
    @overload
    def getVarUpperIIS(self, vars: Mapping[_T, Var]) -> tupledict[_T, int]: ...
    @overload
    def getVarUpperIIS(self, vars: MVar) -> np.ndarray: ...
    @overload
    def getConstrLowerIIS(self, constrs: Constraint) -> int: ...
    @overload
    def getConstrLowerIIS(
        self, constrs: Union[ConstrArray, Iterable[Constraint]]
    ) -> List[int]: ...
    @overload
    def getConstrLowerIIS(
        self, constrs: Mapping[_T, Constraint]
    ) -> tupledict[_T, int]: ...
    @overload
    def getConstrLowerIIS(self, constrs: MConstr) -> np.ndarray: ...
    @overload
    def getConstrUpperIIS(self, constrs: Constraint) -> int: ...
    @overload
    def getConstrUpperIIS(
        self, constrs: Union[ConstrArray, Iterable[Constraint]]
    ) -> List[int]: ...
    @overload
    def getConstrUpperIIS(
        self, constrs: Mapping[_T, Constraint]
    ) -> tupledict[_T, int]: ...
    @overload
    def getConstrUpperIIS(self, constrs: MConstr) -> np.ndarray: ...
    @overload
    def getSOSIIS(self, soss: SOS) -> int: ...
    @overload
    def getSOSIIS(self, soss: Union[SOSArray, Iterable[SOS]]) -> List[int]: ...
    @overload
    def getSOSIIS(self, soss: Mapping[_T, SOS]) -> tupledict[_T, int]: ...
    @overload
    def getIndicatorIIS(self, genconstrs: GenConstr) -> int: ...
    @overload
    def getIndicatorIIS(
        self, genconstrs: Union[GenConstrArray, Iterable[GenConstr]]
    ) -> List[int]: ...
    @overload
    def getIndicatorIIS(
        self, genconstrs: Mapping[_T, GenConstr]
    ) -> tupledict[_T, int]: ...
    def getAttr(self, attrname: str) -> Union[int, float]: ...
    @overload
    def getInfo(
        self,
        infoname: str,
        args: Union[Var, Constraint, NlConstraint, PsdConstraint, QConstraint],
    ) -> float: ...
    @overload
    def getInfo(
        self,
        infoname: str,
        args: Iterable[
            Union[Var, Constraint, NlConstraint, PsdConstraint, QConstraint]
        ],
    ) -> List[float]: ...
    @overload
    def getInfo(
        self,
        infoname: str,
        args: Mapping[
            _T, Union[Var, Constraint, NlConstraint, PsdConstraint, QConstraint]
        ],
    ) -> tupledict[_T, float]: ...
    @overload
    def getInfo(
        self,
        infoname: str,
        args: Union[MVar, MConstr, MQConstr],
    ) -> np.ndarray: ...
    @overload
    def getInfo(
        self,
        infoname: str,
        args: LmiConstraint,
    ) -> List[float]: ...
    @overload
    def getVarType(self, vars: Var) -> str: ...
    @overload
    def getVarType(self, vars: Union[VarArray, Iterable[Var]]) -> List[str]: ...
    @overload
    def getVarType(self, vars: Mapping[_T, Var]) -> tupledict[_T, str]: ...
    @overload
    def getVarType(self, vars: MVar) -> np.ndarray: ...
    def getParam(self, paramname: str) -> Union[int, float]: ...
    def getParamInfo(
        self, paramname: str
    ) -> Union[
        Tuple[str, int, int, int, int], Tuple[str, float, float, float, float]
    ]: ...
    def setBasis(self, varbasis: Sequence[int], constrbasis: Sequence[int]) -> None: ...
    def setSlackBasis(self) -> None: ...
    @overload
    def setVarType(self, vars: Var, vartypes: str) -> None: ...
    @overload
    def setVarType(
        self, vars: Union[VarArray, Iterable[Var]], vartypes: Union[str, Sequence[str]]
    ) -> None: ...
    @overload
    def setVarType(
        self, vars: Mapping[_T, Var], vartypes: Union[str, Mapping[_T, str]]
    ) -> None: ...
    @overload
    def setVarType(
        self,
        vars: MVar,
        vartypes: Union[float, Iterable[float], Iterable[Iterable[float]], np.ndarray],
    ) -> None: ...
    @overload
    def setMipStart(self, vars: Var, startvals: float) -> None: ...
    @overload
    def setMipStart(
        self,
        vars: Union[VarArray, Iterable[Var]],
        startvals: Union[float, Sequence[float]],
    ) -> None: ...
    @overload
    def setMipStart(
        self, vars: Mapping[_T, Var], startvals: Union[float, Mapping[_T, float]]
    ) -> None: ...
    @overload
    def setMipStart(
        self,
        vars: MVar,
        startvals: Union[float, Iterable[float], Iterable[Iterable[float]], np.ndarray],
    ) -> None: ...
    def loadMipStart(self) -> None: ...
    @overload
    def setNlPrimalStart(self, vars: Var, startvals: float) -> None: ...
    @overload
    def setNlPrimalStart(
        self,
        vars: Union[VarArray, Iterable[Var]],
        startvals: Union[float, Sequence[float]],
    ) -> None: ...
    @overload
    def setNlPrimalStart(
        self, vars: Mapping[_T, Var], startvals: Union[float, Mapping[_T, float]]
    ) -> None: ...
    @overload
    def setNlPrimalStart(
        self,
        vars: MVar,
        startvals: Union[float, Iterable[float], Iterable[Iterable[float]], np.ndarray],
    ) -> None: ...
    @overload
    def setInfo(
        self,
        infoname: str,
        args: Union[Var, Constraint, NlConstraint, PsdConstraint],
        newvals: float,
    ) -> None: ...
    @overload
    def setInfo(
        self,
        infoname: str,
        args: Iterable[Union[Var, Constraint, NlConstraint, PsdConstraint]],
        newvals: Union[float, Sequence[float]],
    ) -> None: ...
    @overload
    def setInfo(
        self,
        infoname: str,
        args: Mapping[_T, Union[Var, Constraint, NlConstraint, PsdConstraint]],
        newvals: Union[float, Mapping[_T, float]],
    ) -> None: ...
    @overload
    def setInfo(
        self,
        infoname: str,
        args: Union[MVar, MConstr, MQConstr],
        newvals: Union[float, Iterable[float], Iterable[Iterable[float]], np.ndarray],
    ) -> None: ...
    @overload
    def setNames(
        self,
        args: Union[
            Var,
            PsdVar,
            Constraint,
            GenConstr,
            NlConstraint,
            QConstraint,
            PsdConstraint,
            LmiConstraint,
        ],
        names: str,
    ) -> None: ...
    @overload
    def setNames(
        self,
        args: Iterable[
            Union[
                Var,
                PsdVar,
                Constraint,
                GenConstr,
                NlConstraint,
                QConstraint,
                PsdConstraint,
                LmiConstraint,
            ]
        ],
        names: Union[str, Iterable[str]],
    ) -> None: ...
    @overload
    def setNames(
        self,
        args: Mapping[
            _T,
            Union[
                Var,
                PsdVar,
                Constraint,
                GenConstr,
                NlConstraint,
                QConstraint,
                PsdConstraint,
                LmiConstraint,
            ],
        ],
        names: Union[str, Mapping[_T, str]],
    ) -> None: ...
    @overload
    def setNames(
        self,
        args: Union[
            VarArray,
            ConstrArray,
            GenConstrArray,
            NlConstrArray,
            QConstrArray,
            PsdVarArray,
            PsdConstrArray,
            LmiConstrArray,
        ],
        names: Union[str, Iterable[str]],
    ) -> None: ...
    def setParam(self, paramname: str, newval: Union[int, float]) -> None: ...
    def resetParam(self) -> None: ...
    def read(self, filename: str) -> None: ...
    def readMps(self, filename: str) -> None: ...
    def readLp(self, filename: str) -> None: ...
    def readSdpa(self, filename: str) -> None: ...
    def readCbf(self, filename: str) -> None: ...
    def readBin(self, filename: str) -> None: ...
    def readOrd(self, filename: str) -> None: ...
    def readSol(self, filename: str) -> None: ...
    def readJsonSol(self, filename: str) -> None: ...
    def readBasis(self, filename: str) -> None: ...
    def readMst(self, filename: str) -> None: ...
    def readParam(self, filename: str) -> None: ...
    def readTune(self, filename: str) -> None: ...
    def write(self, filename: str) -> None: ...
    def writeMps(self, filename: str) -> None: ...
    def writeMpsStr(self) -> ProbBuffer: ...
    def writeLp(self, filename: str) -> None: ...
    def writeCbf(self, filename: str) -> None: ...
    def writeBin(self, filename: str) -> None: ...
    def writeOrd(self, filename: str) -> None: ...
    def writeIIS(self, filename: str) -> None: ...
    def writeRelax(self, filename: str) -> None: ...
    def writeSol(self, filename: str) -> None: ...
    def writeJsonSol(self, filename: str) -> None: ...
    def writePoolSol(self, isol: int, filename: str) -> None: ...
    def writeBasis(self, filename: str) -> None: ...
    def writeMst(self, filename: str) -> None: ...
    def writeParam(self, filename: str) -> None: ...
    def writeTuneParam(self, idx: int, filename: str) -> None: ...
    def setCallback(self, cb: ICallback, cbctx: int) -> None: ...
    def setLogFile(self, logfile: str) -> None: ...
    def setLogCallback(self, logcb: Any) -> None: ...
    def solve(self) -> None: ...
    def solveLP(self) -> None: ...
    def computeIIS(self) -> None: ...
    @overload
    def feasRelax(
        self,
        vars: Union[MVar, Var, VarArray, Iterable[Var]],
        lbpen: Union[float, Iterable[float]],
        ubpen: Union[float, Iterable[float]],
        constrs: Union[MConstr, Constraint, ConstrArray, Iterable[Constraint]],
        rhspen: Union[float, Iterable[float]],
        uppen: Optional[Union[float, Iterable[float]]] = None,
    ) -> None: ...
    @overload
    def feasRelax(
        self,
        vars: Mapping[Any, Var],
        lbpen: Mapping[Any, float],
        ubpen: Mapping[Any, float],
        constrs: Mapping[Any, Constraint],
        rhspen: Mapping[Any, float],
        uppen: Optional[Mapping[Any, float]] = None,
    ) -> None: ...
    def feasRelaxN(
        self,
        vars: MVar,
        lbpen: NdArray,
        ubpen: NdArray,
        constrs: MConstr,
        rhspen: NdArray,
        uppen: NdArray = None,
    ) -> None: ...
    def feasRelaxS(self, vrelax: int, crelax: int) -> None: ...
    def tune(self) -> None: ...
    def loadTuenParam(self, idx: int) -> None: ...
    def interrupt(self) -> None: ...
    def remove(
        self,
        args: Union[
            _ModelComponent, Iterable[_ModelComponent], Mapping[Any, _ModelComponent]
        ],
    ) -> None: ...
    def resetAll(self) -> None: ...
    def reset(self) -> None: ...
    def clear(self) -> None: ...
    def clone(self) -> Model: ...

class tupledict(Dict[_T, _U]):
    def clean(self) -> None: ...
    def keys(self) -> tuplelist[_T]: ...
    def values(self) -> List[_U]: ...
    def prod(
        self, coeff: Mapping[Any, float], *pattern: Union[_Scalar, Sequence[_Scalar]]
    ) -> Union[float, LinExpr]: ...
    def select(self, *pattern: Union[_Scalar, Sequence[_Scalar]]) -> List[_U]: ...
    def sum(
        self, *pattern: Union[_Scalar, Sequence[_Scalar]]
    ) -> Union[float, LinExpr]: ...

class tuplelist(List[_T]):
    def clean(self) -> None: ...
    def select(self, *pattern: Union[_Scalar, Sequence[_Scalar]]) -> List[_T]: ...
    def add(self, key: Iterable[_U]) -> None: ...
    def append(self, key: Iterable[_U]) -> None: ...

class MBase(object):
    def dim(self, i: int) -> int: ...
    def item(self) -> object: ...
    def diagonal(self, offset: int = 0, axis1: int = 0, axis2: int = 1) -> MBase: ...
    def expand(self, axis: int = 0) -> MBase: ...
    def flatten(self) -> MBase: ...
    def pick(self, indexes: Union[List[int], List[List[int]], NdArray]) -> MBase: ...
    def repeat(self, repeats: int, axis: int = 0) -> MBase: ...
    def squeeze(self, axis: int = 0) -> MBase: ...
    def stack(self, other: object, axis: int) -> MBase: ...
    def reshape(
        self, shape: Union[int, Tuple[int, ...]], order: str = "C"
    ) -> MBase: ...
    def tolist(self) -> List[object]: ...
    def transpose(self) -> MBase: ...
    @property
    def ndim(self) -> int: ...
    @property
    def shape(self) -> Tuple[int]: ...
    @property
    def size(self) -> int: ...
    @property
    def T(self) -> MBase: ...

class MConstr(MBase):
    def __init__(
        self,
        args: Union[Constraint, Iterable[Constraint], np.ndarray],
        shape: Union[int, Tuple[int]] = None,
    ) -> None: ...
    def getAttr(self, attrname: str) -> np.ndarray: ...
    def setAttr(
        self,
        attrname: str,
        newvalue: Union[float, Iterable[float], Iterable[Iterable[float]], np.ndarray],
    ) -> None: ...
    def getInfo(self, infoname: str) -> np.ndarray: ...
    def setInfo(
        self,
        infoname: str,
        newval: Union[float, Iterable[float], Iterable[Iterable[float]], np.ndarray],
    ) -> None: ...

    # base methods
    def dim(self, i: int) -> int: ...
    def item(self) -> Constraint: ...
    def diagonal(self, offset: int = 0, axis1: int = 0, axis2: int = 1) -> MConstr: ...
    def expand(self, axis: int = 0) -> MConstr: ...
    def flatten(self) -> MConstr: ...
    def pick(self, indexes: Union[List[int], List[List[int]], NdArray]) -> MConstr: ...
    def squeeze(self, axis: int = 0) -> MConstr: ...
    def stack(self, other: Union[Constraint, MConstr], axis: int) -> MConstr: ...
    def hstack(self, other: Union[Constraint, MConstr]) -> MConstr: ...
    def vstack(self, other: Union[Constraint, MConstr]) -> MConstr: ...
    def reshape(
        self, shape: Union[int, Tuple[int, ...]], order: str = "C"
    ) -> MConstr: ...
    def tolist(self) -> List[Constraint]: ...
    def transpose(self) -> MConstr: ...
    def clone(self) -> MConstr: ...
    @property
    def T(self) -> MConstr: ...

class MConstrBuilder:
    def __init__(
        self,
        args: Union[ConstrBuilder, Iterable[ConstrBuilder], np.ndarray],
        shape: Union[int, Tuple[int, ...]] = None,
    ) -> None: ...
    def getExpr(self) -> MLinExpr: ...
    def getSense(self) -> str: ...
    def item(self) -> ConstrBuilder: ...
    def setBuilder(
        self,
        expr: Union[
            MLinExpr, _LinExprLike, Iterable[_LinExprLike], Mapping[Any, _LinExprLike]
        ],
        sense: str,
        rhs: Union[float, MLinExpr, MVar, NdArray],
    ) -> None: ...
    def setRange(
        self,
        expr: Union[
            MLinExpr, _LinExprLike, Iterable[_LinExprLike], Mapping[Any, _LinExprLike]
        ],
        range: float,
    ) -> None: ...

    # read-only info or attributes
    @property
    def expr(self) -> MLinExpr: ...
    @property
    def sense(self) -> str: ...

class MQConstr(MBase):
    def __init__(
        self,
        args: Union[QConstraint, Iterable[QConstraint], np.ndarray],
        shape: Union[int, Tuple[int, ...]] = None,
    ) -> None: ...
    def getAttr(self, attrname: str) -> np.ndarray: ...
    def setAttr(
        self,
        attrname: str,
        newvalue: Union[float, Iterable[float], Iterable[Iterable[float]], np.ndarray],
    ) -> None: ...
    def getInfo(self, infoname: str) -> np.ndarray: ...
    def setInfo(
        self,
        infoname: str,
        newval: Union[float, Iterable[float], Iterable[Iterable[float]], np.ndarray],
    ) -> None: ...

    # base methods
    def dim(self, i: int) -> int: ...
    def item(self) -> QConstraint: ...
    def diagonal(self, offset: int = 0, axis1: int = 0, axis2: int = 1) -> MQConstr: ...
    def expand(self, axis: int = 0) -> MQConstr: ...
    def flatten(self) -> MQConstr: ...
    def pick(self, indexes: Union[List[int], List[List[int]], NdArray]) -> MQConstr: ...
    def squeeze(self, axis: int = 0) -> MQConstr: ...
    def stack(self, other: Union[QConstraint, MQConstr], axis: int) -> MQConstr: ...
    def hstack(self, other: Union[QConstraint, MQConstr]) -> MQConstr: ...
    def vstack(self, other: Union[QConstraint, MQConstr]) -> MQConstr: ...
    def reshape(
        self, shape: Union[int, Tuple[int, ...]], order: str = "C"
    ) -> MQConstr: ...
    def tolist(self) -> List[QConstraint]: ...
    def transpose(self) -> MQConstr: ...
    def clone(self) -> MQConstr: ...
    @property
    def T(self) -> MQConstr: ...

class MQConstrBuilder:
    def __init__(
        self,
        args: Union[
            QConstrBuilder,
            Iterable[QConstrBuilder],
            Iterable[Iterable[QConstrBuilder]],
            np.ndarray,
        ],
        shape: Union[int, Tuple[int, ...]] = None,
    ) -> None: ...
    def getQuadExpr(self) -> MQuadExpr: ...
    def getSense(self) -> str: ...
    def item(self) -> QConstrBuilder: ...
    def setBuilder(
        self,
        expr: Union[
            MQuadExpr,
            _QuadExprLike,
            Iterable[_QuadExprLike],
            Mapping[Any, _QuadExprLike],
        ],
        sense: str,
        rhs: Union[float, MLinExpr, MVar, MQuadExpr, NdArray],
    ) -> None: ...

    # read-only info or attributes
    @property
    def qexpr(self) -> MQuadExpr: ...
    @property
    def sense(self) -> str: ...

class MPsdConstr(MBase):
    def __init__(
        self,
        args: Union[PsdConstraint, Iterable[PsdConstraint]],
        shape: Union[int, Tuple[int, ...]] = None,
    ) -> None: ...
    def getAttr(self, attrname: str) -> Union[NdArray, np.ndarray]: ...
    def setAttr(
        self,
        attrname: str,
        newval: Union[float, Iterable[float], Iterable[Iterable[float]], NdArray],
    ) -> None: ...
    def getInfo(self, infoname: str) -> Union[NdArray, np.ndarray]: ...
    def setInfo(
        self,
        infoname: str,
        newval: Union[float, Iterable[float], Iterable[Iterable[float]], NdArray],
    ) -> None: ...

    # base methods
    def dim(self, i: int) -> int: ...
    def item(self) -> PsdConstraint: ...
    def diagonal(
        self, offset: int = 0, axis1: int = 0, axis2: int = 1
    ) -> MPsdConstr: ...
    def expand(self, axis: int = 0) -> MPsdConstr: ...
    def flatten(self) -> MPsdConstr: ...
    def pick(
        self, indexes: Union[List[int], List[List[int]], NdArray]
    ) -> MPsdConstr: ...
    def squeeze(self, axis: int = 0) -> MPsdConstr: ...
    def stack(
        self, other: Union[PsdConstraint, MPsdConstr], axis: int
    ) -> MPsdConstr: ...
    def hstack(self, other: Union[PsdConstraint, MPsdConstr]) -> MPsdConstr: ...
    def vstack(self, other: Union[PsdConstraint, MPsdConstr]) -> MPsdConstr: ...
    def reshape(
        self, shape: Union[int, Tuple[int, ...]], order: str = "C"
    ) -> MPsdConstr: ...
    def tolist(self) -> List[PsdConstraint]: ...
    def transpose(self) -> MPsdConstr: ...
    def clone(self) -> MPsdConstr: ...
    @property
    def T(self) -> MPsdConstr: ...

class MPsdConstrBuilder:
    def __init__(self, args: Union[int, Tuple[int, ...]] = None) -> None: ...
    def getPsdExpr(self) -> MPsdExpr: ...
    def getRange(self) -> float: ...
    def getSense(self) -> str: ...
    def setBuilder(
        self,
        expr: Union[
            MPsdExpr, _PsdExprLike, Iterable[_PsdExprLike], Mapping[Any, _PsdExprLike]
        ],
        sense: str,
        rhs: Union[float, MLinExpr, MVar, MPsdExpr, NdArray],
    ) -> None: ...
    def setRange(
        self,
        expr: Union[
            MPsdExpr, _PsdExprLike, Iterable[_PsdExprLike], Mapping[Any, _PsdExprLike]
        ],
        range: float,
    ) -> None: ...

    # read-only info or attributes
    @property
    def psdexpr(self) -> MPsdExpr: ...
    @property
    def range(self) -> float: ...
    @property
    def sense(self) -> str: ...

class MLinExpr(MBase):
    @overload
    def __init__(
        self,
        args: Union[_LinExprLike, Iterable[_LinExprLike], np.ndarray],
        shape: Union[int, Tuple[int, ...]] = None,
    ) -> None: ...
    @overload
    def __init__(
        self, args: Union[MVar, MLinExpr], shape: Union[int, Tuple[int, ...]] = None
    ) -> None: ...
    def __getitem__(
        self, idx: Union[int, slice, Tuple[Union[int, slice], ...]]
    ) -> MLinExpr: ...
    def addTerms(self, vars: MVar, coeffs: Union[float, List[float]]) -> None: ...
    def addMLinExpr(self, exprs: MLinExpr, mult: float = 1.0) -> None: ...
    def zeros(shape) -> MLinExpr: ...
    def clear(self) -> None: ...
    def clone(self) -> MLinExpr: ...
    def getValue(self) -> NdArray: ...
    def sum(self, axis: int = None) -> MLinExpr: ...

    # base methods
    def dim(self, i: int) -> int: ...
    def item(self) -> LinExpr: ...
    def diagonal(self, offset: int = 0, axis1: int = 0, axis2: int = 1) -> MLinExpr: ...
    def expand(self, axis: int = 0) -> MLinExpr: ...
    def flatten(self) -> MLinExpr: ...
    def pick(self, indexes: Union[List[int], List[List[int]], NdArray]) -> MLinExpr: ...
    def squeeze(self, axis: int = 0) -> MLinExpr: ...
    def stack(self, other: Union[_LinExprLike, MLinExpr], axis: int) -> MLinExpr: ...
    def hstack(self, other: Union[_LinExprLike, MLinExpr]) -> MLinExpr: ...
    def vstack(self, other: Union[_LinExprLike, MLinExpr]) -> MLinExpr: ...
    def repeat(self, repeats: int, axis: int = 0) -> MLinExpr: ...
    def reshape(
        self, shape: Union[int, Tuple[int, ...]], order: str = "C"
    ) -> MLinExpr: ...
    def tolist(self) -> List[LinExpr]: ...
    def transpose(self) -> MLinExpr: ...
    @property
    def T(self) -> MLinExpr: ...
    @overload
    def __add__(self, other: _LinExprLike) -> MLinExpr: ...
    @overload
    def __add__(self, other: Union[MVar, MLinExpr]) -> MLinExpr: ...
    @overload
    def __add__(self, other: List[Any]) -> MLinExpr: ...
    @overload
    def __add__(self, other: MQuadExpr) -> MQuadExpr: ...
    @overload
    def __radd__(self, other: np.ndarray) -> MLinExpr: ...
    @overload
    def __radd__(self, other: List[Any]) -> MLinExpr: ...
    @overload
    def __sub__(self, other: _LinExprLike) -> MLinExpr: ...
    @overload
    def __sub__(self, other: Union[MVar, MLinExpr]) -> MLinExpr: ...
    @overload
    def __sub__(self, other: MQuadExpr) -> MQuadExpr: ...
    @overload
    def __rsub__(self, other: np.ndarray) -> MLinExpr: ...
    @overload
    def __rsub__(self, other: List[Any]) -> MLinExpr: ...
    @overload
    def __mul__(self, other: _LinExprLike) -> MQuadExpr: ...
    @overload
    def __mul__(self, other: Union[MVar, MLinExpr]) -> MQuadExpr: ...
    @overload
    def __matmul__(
        self, other: Union[_LinExprLike, Iterable[_LinExprLike], np.ndarray]
    ) -> MQuadExpr: ...
    @overload
    def __matmul__(self, other: Union[MVar, MLinExpr]) -> MQuadExpr: ...
    @overload
    def __rmatmul__(self, other: np.ndarray) -> MLinExpr: ...
    @overload
    def __rmatmul__(self, other: float) -> MLinExpr: ...
    @overload
    def __eq__(self, other: np.ndarray) -> MConstrBuilder: ...
    @overload
    def __eq__(self, other: _LinExprLike) -> MConstrBuilder: ...
    @overload
    def __eq__(self, other: Union[MVar, MLinExpr]) -> MConstrBuilder: ...
    @overload
    def __eq__(self, other: Union[List[Any], List[List[Any]]]) -> MConstrBuilder: ...
    @overload
    def __eq__(self, other: MQuadExpr) -> MQConstrBuilder: ...
    @overload
    def __ge__(self, other: np.ndarray) -> MConstrBuilder: ...
    @overload
    def __ge__(self, other: _LinExprLike) -> MConstrBuilder: ...
    @overload
    def __ge__(self, other: Union[MVar, MLinExpr]) -> MConstrBuilder: ...
    @overload
    def __ge__(self, other: Union[List[Any], List[List[Any]]]) -> MConstrBuilder: ...
    @overload
    def __ge__(self, other: MQuadExpr) -> MQConstrBuilder: ...
    @overload
    def __le__(self, other: np.ndarray) -> MConstrBuilder: ...
    @overload
    def __le__(self, other: _LinExprLike) -> MConstrBuilder: ...
    @overload
    def __le__(self, other: Union[MVar, MLinExpr]) -> MConstrBuilder: ...
    @overload
    def __le__(self, other: Union[List[Any], List[List[Any]]]) -> MConstrBuilder: ...
    @overload
    def __le__(self, other: MQuadExpr) -> MQConstrBuilder: ...

class MQuadExpr(MBase):
    @overload
    def __init__(
        self,
        args: Union[_QuadExprLike, Iterable[_QuadExprLike], np.ndarray],
        shape: Union[int, Tuple[int, ...]] = None,
    ) -> None: ...
    @overload
    def __init__(
        self,
        args: Union[MVar, MLinExpr, MQuadExpr],
        shape: Union[int, Tuple[int, ...]] = None,
    ) -> None: ...
    def __getitem__(
        self, idx: Union[int, slice, Tuple[Union[int, slice], ...]]
    ) -> MQuadExpr: ...
    def addTerms(self, vars: MVar, coeffs: Union[float, List[float]]) -> None: ...
    def addMLinExpr(self, exprs: MLinExpr, mult: float = 1.0) -> None: ...
    def addMQuadExpr(self, exprs: MQuadExpr, mult: float = 1.0) -> None: ...
    def zeros(shape) -> MQuadExpr: ...
    def clear(self) -> None: ...
    def clone(self) -> MQuadExpr: ...
    def getValue(self) -> NdArray: ...
    def sum(self, axis: int = None) -> MQuadExpr: ...

    # base methods
    def dim(self, i: int) -> int: ...
    def item(self) -> QuadExpr: ...
    def diagonal(
        self, offset: int = 0, axis1: int = 0, axis2: int = 1
    ) -> MQuadExpr: ...
    def expand(self, axis: int = 0) -> MQuadExpr: ...
    def flatten(self) -> MQuadExpr: ...
    def pick(
        self, indexes: Union[List[int], List[List[int]], NdArray]
    ) -> MQuadExpr: ...
    def squeeze(self, axis: int = 0) -> MQuadExpr: ...
    def stack(self, other: Union[_QuadExprLike, MQuadExpr], axis: int) -> MQuadExpr: ...
    def hstack(self, other: Union[_QuadExprLike, MQuadExpr]) -> MQuadExpr: ...
    def vstack(self, other: Union[_QuadExprLike, MQuadExpr]) -> MQuadExpr: ...
    def repeat(self, repeats: int, axis: int = 0) -> MQuadExpr: ...
    def reshape(
        self, shape: Union[int, Tuple[int, ...]], order: str = "C"
    ) -> MQuadExpr: ...
    def tolist(self) -> List[QuadExpr]: ...
    def transpose(self) -> MQuadExpr: ...
    @property
    def T(self) -> MQuadExpr: ...
    @overload
    def __add__(self, other: _QuadExprLike) -> MQuadExpr: ...
    @overload
    def __add__(self, other: Union[MVar, MLinExpr, MQuadExpr]) -> MQuadExpr: ...
    @overload
    def __sub__(self, other: _QuadExprLike) -> MQuadExpr: ...
    @overload
    def __sub__(self, other: Union[MVar, MLinExpr, MQuadExpr]) -> MQuadExpr: ...
    def __mul__(self, other: float) -> MQuadExpr: ...
    def __matmul__(
        self, other: Union[float, Iterable[float], np.ndarray]
    ) -> MQuadExpr: ...
    @overload
    def __eq__(self, other: _QuadExprLike) -> MQConstrBuilder: ...
    @overload
    def __eq__(self, other: Union[MVar, MLinExpr, MQuadExpr]) -> MQConstrBuilder: ...
    @overload
    def __ge__(self, other: _QuadExprLike) -> MQConstrBuilder: ...
    @overload
    def __ge__(self, other: Union[MVar, MLinExpr, MQuadExpr]) -> MQConstrBuilder: ...
    @overload
    def __le__(self, other: _QuadExprLike) -> MQConstrBuilder: ...
    @overload
    def __le__(self, other: Union[MVar, MLinExpr, MQuadExpr]) -> MQConstrBuilder: ...

class MPsdExpr(MBase):
    @overload
    def __init__(
        self,
        args: Union[_PsdExprLike, Iterable[_PsdExprLike], Mapping[Any, _PsdExprLike]],
        shape: Union[int, Tuple[int, ...]] = None,
    ) -> None: ...
    @overload
    def __init__(
        self,
        args: Union[MVar, MLinExpr, MPsdExpr],
        shape: Union[int, Tuple[int, ...]] = None,
    ) -> None: ...
    def __getitem__(
        self, idx: Union[int, slice, Tuple[Union[int, slice], ...]]
    ) -> MPsdExpr: ...
    def addTerm(
        self, var: Union[Var, PsdVar], mat: Union[float, SymMatrix, SymMatExpr]
    ) -> None: ...
    def addTerms(self, vars: MVar, coeffs: Union[float, List[float]]) -> None: ...
    def addLinExpr(
        self, expr: Union[LinExpr, ExprBuilder], mult: float = 1.0
    ) -> None: ...
    def addPsdExpr(self, expr: PsdExpr, mult: float = 1.0) -> None: ...
    def addMLinExpr(self, exprs: MLinExpr, mult: float = 1.0) -> None: ...
    def addMPsdExpr(self, exprs: MPsdExpr, mult: float = 1.0) -> None: ...
    def zeros(shape) -> MPsdExpr: ...
    def clear(self) -> None: ...
    def clone(self) -> MPsdExpr: ...
    def getValue(self) -> NdArray: ...
    def sum(self, axis: int = None) -> MPsdExpr: ...

    # base methods
    def dim(self, i: int) -> int: ...
    def item(self) -> PsdExpr: ...
    def diagonal(self, offset: int = 0, axis1: int = 0, axis2: int = 1) -> MPsdExpr: ...
    def expand(self, axis: int = 0) -> MPsdExpr: ...
    def flatten(self) -> MPsdExpr: ...
    def pick(self, indexes: Union[List[int], List[List[int]], NdArray]) -> MPsdExpr: ...
    def squeeze(self, axis: int = 0) -> MPsdExpr: ...
    def stack(self, other: Union[_PsdExprLike, MPsdExpr], axis: int) -> MPsdExpr: ...
    def hstack(self, other: Union[_PsdExprLike, MPsdExpr]) -> MPsdExpr: ...
    def vstack(self, other: Union[_PsdExprLike, MPsdExpr]) -> MPsdExpr: ...
    def repeat(self, repeats: int, axis: int = 0) -> MPsdExpr: ...
    def reshape(
        self, shape: Union[int, Tuple[int, ...]], order: str = "C"
    ) -> MPsdExpr: ...
    def tolist(self) -> List[PsdExpr]: ...
    def transpose(self) -> MPsdExpr: ...
    @property
    def T(self) -> MPsdExpr: ...
    @overload
    def __add__(self, other: _PsdExprLike) -> MPsdExpr: ...
    @overload
    def __add__(self, other: Union[MVar, MLinExpr, MPsdExpr]) -> MPsdExpr: ...
    @overload
    def __sub__(self, other: _PsdExprLike) -> MPsdExpr: ...
    @overload
    def __sub__(self, other: Union[MVar, MLinExpr, MPsdExpr]) -> MPsdExpr: ...
    def __mul__(self, other: float) -> MPsdExpr: ...
    def __matmul__(self, other: Union[float, Iterable[float], NdArray]) -> MPsdExpr: ...
    @overload
    def __eq__(self, other: _PsdExprLike) -> MPsdConstrBuilder: ...
    @overload
    def __eq__(self, other: Union[MVar, MLinExpr, MPsdExpr]) -> MPsdConstrBuilder: ...
    @overload
    def __ge__(self, other: _PsdExprLike) -> MPsdConstrBuilder: ...
    @overload
    def __ge__(self, other: Union[MVar, MLinExpr, MPsdExpr]) -> MPsdConstrBuilder: ...
    @overload
    def __le__(self, other: _PsdExprLike) -> MPsdConstrBuilder: ...
    @overload
    def __le__(self, other: Union[MVar, MLinExpr, MPsdExpr]) -> MPsdConstrBuilder: ...

class MVar(MBase):
    # mutable info or params
    # TODO: add more if needed
    lb: Union[float, np.ndarray[float]] = ...
    LB: Union[float, np.ndarray[float]] = ...
    name: Union[str, np.ndarray[str]] = ...
    obj: Union[float, np.ndarray[float]] = ...
    OBJ: Union[float, np.ndarray[float]] = ...
    ub: Union[float, np.ndarray[float]] = ...
    UB: Union[float, np.ndarray[float]] = ...
    vtype: Union[str, np.ndarray[str]] = ...
    VType: Union[str, np.ndarray[str]] = ...

    def __init__(
        self,
        args: Union[Var, Iterable[Var], Iterable[Iterable[Var]], np.ndarray],
        shape: Union[int, Tuple[int, ...]] = None,
    ) -> None: ...
    def __getitem__(
        self, idx: Union[int, slice, Tuple[Union[int, slice], ...]]
    ) -> MVar: ...
    def clone(self) -> MVar: ...
    def fromlist(
        vars: Union[
            Iterable[Var],
            Iterable[Iterable[Var]],
            Iterable[Iterable[Iterable[Var]]],
            np.ndarray,
        ],
    ) -> MVar: ...
    def fromvar(var: Var) -> MVar: ...
    def getAttr(self, attrname: str) -> np.ndarray: ...
    def setAttr(
        self,
        attrname: str,
        newvalue: Union[float, Iterable[float], Iterable[Iterable[float]], np.ndarray],
    ) -> None: ...
    def getInfo(self, infoname: str) -> np.ndarray: ...
    def setInfo(
        self,
        infoname: str,
        newval: Union[float, Iterable[float], Iterable[Iterable[float]], np.ndarray],
    ) -> None: ...
    def sum(self, axis: int = None) -> MLinExpr: ...

    # base methods
    def dim(self, i: int) -> int: ...
    def item(self) -> Var: ...
    def diagonal(self, offset: int = 0, axis1: int = 0, axis2: int = 1) -> MVar: ...
    def expand(self, axis: int = 0) -> MVar: ...
    def flatten(self) -> MVar: ...
    def pick(self, indexes: Union[List[int], List[List[int]], NdArray]) -> MVar: ...
    def squeeze(self, axis: int = 0) -> MVar: ...
    def stack(self, other: Union[Var, MVar], axis: int) -> MVar: ...
    def hstack(self, other: Union[Var, MVar]) -> MVar: ...
    def vstack(self, other: Union[Var, MVar]) -> MVar: ...
    def repeat(self, repeats: int, axis: int = 0) -> MVar: ...
    def reshape(self, shape: Union[int, Tuple[int, ...]], order: str = "C") -> MVar: ...
    def tolist(self) -> List[Var]: ...
    def transpose(self) -> MVar: ...
    @property
    def T(self) -> MVar: ...
    @overload
    def __add__(self, other: _LinExprLike) -> MLinExpr: ...
    @overload
    def __add__(self, other: Union[MVar, MLinExpr]) -> MLinExpr: ...
    @overload
    def __add__(self, other: List[Any]) -> MLinExpr: ...
    @overload
    def __add__(self, other: MQuadExpr) -> MQuadExpr: ...
    @overload
    def __radd__(self, other: _LinExprLike) -> MLinExpr: ...
    @overload
    def __radd__(self, other: List[Any]) -> MLinExpr: ...
    @overload
    def __sub__(self, other: _LinExprLike) -> MLinExpr: ...
    @overload
    def __sub__(self, other: Union[MVar, MLinExpr]) -> MLinExpr: ...
    @overload
    def __sub__(self, other: MQuadExpr) -> MQuadExpr: ...
    @overload
    def __rsub__(self, other: _LinExprLike) -> MLinExpr: ...
    @overload
    def __mul__(self, other: _LinExprLike) -> MQuadExpr: ...
    @overload
    def __mul__(self, other: Union[MVar, MLinExpr]) -> MQuadExpr: ...
    @overload
    def __rmul__(self, other: float) -> MLinExpr: ...
    @overload
    def __rmul__(self, other: Union[Var, ExprBuilder, LinExpr]) -> MQuadExpr: ...
    @overload
    def __matmul__(self, other: float) -> MLinExpr: ...
    @overload
    def __matmul__(self, other: np.ndarray) -> MLinExpr: ...
    @overload
    def __matmul__(
        self, other: Union[_LinExprLike, Iterable[_LinExprLike]]
    ) -> MQuadExpr: ...
    @overload
    def __matmul__(self, other: Union[MVar, MLinExpr]) -> MQuadExpr: ...
    @overload
    def __rmatmul__(self, other: np.ndarray) -> MLinExpr: ...
    @overload
    def __rmatmul__(self, other: float) -> MLinExpr: ...
    @overload
    def __neg__(self) -> MLinExpr: ...
    @overload
    def __eq__(self, other: np.ndarray) -> MConstrBuilder: ...
    @overload
    def __eq__(self, other: _LinExprLike) -> MConstrBuilder: ...
    @overload
    def __eq__(self, other: Union[MVar, MLinExpr]) -> MConstrBuilder: ...
    @overload
    def __eq__(self, other: Union[List[Any], List[List[Any]]]) -> MConstrBuilder: ...
    @overload
    def __eq__(self, other: MQuadExpr) -> MQConstrBuilder: ...
    @overload
    def __ge__(self, other: np.ndarray) -> MConstrBuilder: ...
    @overload
    def __ge__(self, other: _LinExprLike) -> MConstrBuilder: ...
    @overload
    def __ge__(self, other: Union[MVar, MLinExpr]) -> MConstrBuilder: ...
    @overload
    def __ge__(self, other: Union[List[Any], List[List[Any]]]) -> MConstrBuilder: ...
    @overload
    def __ge__(self, other: MQuadExpr) -> MQConstrBuilder: ...
    @overload
    def __le__(self, other: np.ndarray) -> MConstrBuilder: ...
    @overload
    def __le__(self, other: _LinExprLike) -> MConstrBuilder: ...
    @overload
    def __le__(self, other: Union[MVar, MLinExpr]) -> MConstrBuilder: ...
    @overload
    def __le__(self, other: Union[List[Any], List[List[Any]]]) -> MConstrBuilder: ...
    @overload
    def __le__(self, other: MQuadExpr) -> MQConstrBuilder: ...

class View:
    def __init__(self) -> None: ...
    def apply(
        self, idx: Union[int, slice, Tuple[Union[int, slice], ...]] = None
    ) -> View: ...
    def addFull(self) -> View: ...
    def addScalar(self, n: int) -> View: ...
    def addSlice(self, start: int, end: int, step: int = None) -> View: ...

class Shape:
    def __init__(self, args: Union[int, Tuple[int, ...]] = None) -> None: ...
    def __getitem__(self, idx: int) -> int: ...
    def __iter__(self) -> int: ...
    def __len__(self) -> int: ...
    def expand(self, axis: int = 0) -> Shape: ...
    def expandto(self, ndim: int) -> Shape: ...
    def squeeze(self, axis: int = 0) -> Shape: ...
    @property
    def size(self) -> int: ...
    @property
    def strides(self) -> Tuple[int, ...]: ...
    @property
    def starts(self) -> Tuple[int, ...]: ...

class NdArray:
    def __init__(
        self,
        args: Union[_Scalar, Iterable[_Scalar], Iterable[Iterable[_Scalar]]] = None,
        dtype: type = None,
        shape: Union[int, Tuple[int, ...]] = None,
    ) -> None: ...
    def __getitem__(
        self, idx: Union[int, slice, Tuple[Union[int, slice], ...]]
    ) -> NdArray: ...
    def __iter__(self) -> Union[int, float]: ...
    def dim(self, i: int) -> None: ...
    def item(self) -> object: ...
    def diagonal(self, offset: int = 0, axis1: int = 0, axis2: int = 1) -> MBase: ...
    def expand(self, axis: int = 0) -> NdArray: ...
    def fill(self, value: float) -> None: ...
    def flatten(self) -> NdArray: ...
    def pick(self, indexes: Union[List[int], List[List[int]], NdArray]) -> NdArray: ...
    def squeeze(self, axis: int = 0) -> NdArray: ...
    def stack(self, other: Union[_Scalar, NdArray], axis: int) -> NdArray: ...
    def hstack(self, other: Union[_Scalar, NdArray]) -> NdArray: ...
    def vstack(self, other: Union[_Scalar, NdArray]) -> NdArray: ...
    def sum(self, axis: int = 0) -> NdArray: ...
    def repeat(self, repeats: int, axis: int = 0) -> NdArray: ...
    def reshape(
        self, shape: Union[int, Tuple[int, ...]], order: str = "C"
    ) -> NdArray: ...
    def tolist(self) -> List[Union[int, float]]: ...
    def tonumpy(self) -> np.ndarray: ...
    def transpose(self) -> NdArray: ...
    @property
    def dtype(self) -> type: ...
    @property
    def ndim(self) -> int: ...
    @property
    def shape(self) -> Tuple[int]: ...
    @property
    def size(self) -> int: ...
    @property
    def T(self) -> NdArray: ...
    @overload
    def __add__(self, other: Union[int, List[int], List[List[int]]]) -> NdArray: ...
    @overload
    def __add__(
        self, other: Union[float, List[float], List[List[float]]]
    ) -> NdArray: ...
    @overload
    def __add__(self, other: NdArray) -> NdArray: ...
    @overload
    def __sub__(self, other: Union[int, List[int], List[List[int]]]) -> NdArray: ...
    @overload
    def __sub__(
        self, other: Union[float, List[float], List[List[float]]]
    ) -> NdArray: ...
    @overload
    def __sub__(self, other: NdArray) -> NdArray: ...
    @overload
    def __mul__(self, other: Union[int, List[int], List[List[int]]]) -> NdArray: ...
    @overload
    def __mul__(
        self, other: Union[float, List[float], List[List[float]]]
    ) -> NdArray: ...
    @overload
    def __mul__(self, other: NdArray) -> NdArray: ...
    @overload
    def __matmul__(self, other: Union[int, List[int], List[List[int]]]) -> NdArray: ...
    @overload
    def __matmul__(
        self, other: Union[float, List[float], List[List[float]]]
    ) -> NdArray: ...
    @overload
    def __matmul__(self, other: NdArray) -> NdArray: ...

class nl:
    @staticmethod
    def sum(op1: Union[Iterable[NlExpr], Mapping[_T, NlExpr]]) -> NlExpr: ...
    @staticmethod
    def sum(op1: _NlExprLike, op2: _NlExprLike, op3: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def sum(op1: _NlExprLike, op2: _NlExprLike, op3: _NlExprLike, op4: _NlExprLike
    ) -> NlExpr: ...
    @staticmethod
    def atan2(y: _NlExprLike, x: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def pow(base: _NlExprLike, expo: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def sqrt(expr: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def exp(expr: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def log(expr: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def log10(expr: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def neg(expr: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def abs(expr: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def floor(expr: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def ceil(expr: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def cos(expr: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def sin(expr: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def tan(expr: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def cosh(expr: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def sinh(expr: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def tanh(expr: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def acos(expr: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def asin(expr: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def atan(expr: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def acosh(expr: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def asinh(expr: _NlExprLike) -> NlExpr: ...
    @staticmethod
    def atanh(expr: _NlExprLike) -> NlExpr: ...
