#! /usr/bin/python
#
# Install script of Python API for the Cardinal Optimizer
#
from __future__ import print_function

import os
import sys
import shutil
import platform
import subprocess

from distutils.core import setup, Command
from distutils.command.clean import clean
from distutils.command.install import install

class CoptpyClean(Command):
  def initialize_options(self):
    self.setup_dir = None
    self.force = 1
  
  def finalize_options(self):
    self.setup_dir = os.path.dirname(os.path.realpath(__file__))
  
  def run(self):
    if os.getcwd().lower() != self.setup_dir.lower():
      raise Exception("Please install package from 'setup.py' directory")

    build_dir = os.path.join(os.getcwd(), "build")
    if os.path.exists(build_dir):
      print("Remove build directory '{}'".format(build_dir))
      shutil.rmtree(build_dir)

class CoptpyInstall(install):
  def install_dependency(self):
    coptpydir = os.path.join(self.install_lib, "coptpy")

    if sys.platform == "win32":
      coptccdir = os.path.join(os.path.dirname(os.path.abspath(__file__)), "deps")
      shutil.copy(os.path.join(coptccdir, "copt_python.dll"), coptpydir)
    elif sys.platform == "linux" or sys.platform == "linux2":
      coptccdir = os.path.join(os.path.dirname(os.path.abspath(__file__)), "deps")
      shutil.copy(os.path.join(coptccdir, "libcopt_python.so"), coptpydir)
    elif sys.platform == "darwin":
      coptccdir = os.path.join(os.path.dirname(os.path.abspath(__file__)), "deps")
      shutil.copy(os.path.join(coptccdir, "libcopt_python.dylib"), coptpydir)
      subprocess.call(['xattr', '-dr', 'com.apple.quarantine', coptpydir])

  def run(self):
    install.run(self)
    self.execute(self.install_dependency, (), msg="Install dependency libraries for 'coptpy'")
    act_clean = CoptpyClean(self.distribution)
    act_clean.finalize_options()
    act_clean.run()

def get_pyversion():
  pyver = sys.version_info

  if sys.platform == "darwin" and platform.machine() == "arm64":
    if pyver.major == 3 and pyver.minor in [8, 9, 10, 11, 12, 13]:
      print("Install 'coptpy' for Python {0}.{1}".format(pyver.major, pyver.minor))
    else:
      raise RuntimeError("Unsupported Python {0}.{1} for 'coptpy'".format(pyver.major, pyver.minor))
  else:
    if pyver.major == 3 and pyver.minor in [7, 8, 9, 10, 11, 12, 13]:
      print("Install 'coptpy' for Python {0}.{1}".format(pyver.major, pyver.minor))
    else:
      raise RuntimeError("Unsupported Python {0}.{1} for 'coptpy'".format(pyver.major, pyver.minor))

  return "{}{}".format(pyver.major, pyver.minor)

def get_pkgdata():
  pkgpath = os.path.join(get_pyversion(), "coptpy")
  pyver = sys.version_info
  if pyver.major > 2:
    pkgfile = ['__init__.py', 'coptcore.pyi', 'py.typed']
  else:
    raise FileNotFoundError("Support for python27 is removed from coptpy")

  if sys.platform == "win32":
    pkgfile.append("coptpywrap.pyd")
    pkgfile.append("coptcore.pyd")
  elif sys.platform == "linux" or sys.platform == "linux2" or sys.platform == "darwin":
    pkgfile.append("coptpywrap.so")
    pkgfile.append("coptcore.so")
  else:
    raise RuntimeError("Unsupported operating system '{}'".format(sys.platform))

  return pkgpath, pkgfile

def install_coptpy():
  pkgpath, pkgfile = get_pkgdata()

  setup(name="coptpy",
        version="8.0.1",
        description="The Python interface of the Cardinal Optimizer",
        author="Cardinal Operations, LLC",
        author_email="coptsales@shanshu.ai",
        url="www.shanshu.ai",
        license="""Cardinal Operations, LLC
         End User License Agreement""",
        packages=["coptpy"],
        package_dir={"coptpy": pkgpath},
        package_data={"coptpy": pkgfile},
        cmdclass={"install": CoptpyInstall, "clean": CoptpyClean}
        )

if __name__ == "__main__":
  install_coptpy()
